<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\BabysitterProfile;
use App\Models\User;
use App\Models\Role;
use App\Models\OccupationCategory;

class BabysitterProfilesTableSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get babysitter role
        $babysitterRole = Role::where('role_name', 'babysitter')->first();

        if (!$babysitterRole) {
            $this->command->error('Babysitter role not found. Please run RolesTableSeeder first.');
            return;
        }

        // Get occupation categories
        $categories = OccupationCategory::all();
        if ($categories->isEmpty()) {
            $this->command->error('No occupation categories found. Please run OccupationCategoriesTableSeeder first.');
            return;
        }

        $babysitterUsers = User::where('role_id', $babysitterRole->role_id)->get();

        if ($babysitterUsers->isEmpty()) {
            $this->command->error('No babysitter users found. Please create some babysitter users first.');
            return;
        }

        $babysitterProfiles = [
            [
                'user_id' => $babysitterUsers[0]->user_id,
                'occupation_category_id' => $categories->where('category_type', 'nanny')->first()->category_id,
                'hourly_rate' => 22.00,
                'hourly_rates' => json_encode(['CAD' => 22.00, 'USD' => 16.50]),
                'experience_years' => 5,
                'availability_type' => 'full_time',
                'immediate_availability' => true,
                'service_range_km' => 25,
                'willing_to_travel' => true,
                'has_vehicle' => true,
                'has_drivers_license' => true,
                'about_me' => 'I am a certified childcare professional with 5 years of experience working with children of all ages. I have first aid and CPR certification.',
                'why_babysit' => 'I love working with children and helping them learn and grow. Every child is unique and I enjoy discovering their individual personalities.',
                'is_profile_public' => true,
                'is_verified' => true,
                'verification_status' => 'verified',
                'total_reviews' => 12,
                'average_rating' => 4.8,
                'completed_jobs' => 45,
                'response_rate' => 95.5,
                'response_time_minutes' => 15,
                'search_tags' => json_encode(['certified', 'first_aid', 'cpr', 'experienced', 'reliable', 'fun']),
            ],
            [
                'user_id' => $babysitterUsers[1]->user_id ?? $babysitterUsers[0]->user_id,
                'occupation_category_id' => $categories->where('category_type', 'nanny')->first()->category_id,
                'hourly_rate' => 18.00,
                'hourly_rates' => json_encode(['CAD' => 18.00, 'USD' => 13.50]),
                'experience_years' => 2,
                'availability_type' => 'part_time',
                'immediate_availability' => false,
                'service_range_km' => 15,
                'willing_to_travel' => false,
                'has_vehicle' => false,
                'has_drivers_license' => true,
                'about_me' => 'I am a university student studying early childhood education. I have experience with children aged 2-10 and can help with homework.',
                'why_babysit' => 'I enjoy spending time with children and gaining practical experience for my future career in education.',
                'is_profile_public' => true,
                'is_verified' => true,
                'verification_status' => 'verified',
                'total_reviews' => 8,
                'average_rating' => 4.5,
                'completed_jobs' => 22,
                'response_rate' => 88.0,
                'response_time_minutes' => 30,
                'search_tags' => json_encode(['student', 'homework_help', 'creative', 'patient', 'educational']),
            ],
            [
                'user_id' => $babysitterUsers[2]->user_id ?? $babysitterUsers[0]->user_id,
                'occupation_category_id' => $categories->where('category_type', 'special_needs_care')->first()->category_id,
                'hourly_rate' => 25.00,
                'hourly_rates' => json_encode(['CAD' => 25.00, 'USD' => 18.75]),
                'experience_years' => 8,
                'availability_type' => 'flexible',
                'immediate_availability' => true,
                'service_range_km' => 30,
                'willing_to_travel' => true,
                'has_vehicle' => true,
                'has_drivers_license' => true,
                'about_me' => 'Specialized in special needs care with 8 years of experience. Trained in autism spectrum disorders, ADHD, and developmental disabilities.',
                'why_babysit' => 'I am passionate about providing quality care for children with special needs and helping them reach their full potential.',
                'is_profile_public' => true,
                'is_verified' => true,
                'verification_status' => 'verified',
                'total_reviews' => 25,
                'average_rating' => 4.9,
                'completed_jobs' => 120,
                'response_rate' => 98.0,
                'response_time_minutes' => 10,
                'search_tags' => json_encode(['special_needs', 'autism', 'adhd', 'experienced', 'compassionate', 'professional']),
            ]
        ];

        foreach ($babysitterProfiles as $profile) {
            // Check if user exists and doesn't already have a profile
            $user = User::find($profile['user_id']);
            if ($user && !BabysitterProfile::where('user_id', $profile['user_id'])->exists()) {
                BabysitterProfile::create($profile);
                $this->command->info("Created babysitter profile for user: {$user->first_name} {$user->last_name}");
            }
        }

        $this->command->info('Babysitter profiles seeded successfully!');
    }
}
