<?php

namespace App\Notifications;

use App\Models\Payment;
use App\Models\Booking;
use App\Models\CommissionEarning;
use App\Models\Payout;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class SuperAdminPaymentNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $payment;
    public $booking;
    public $commissionEarning;
    public $payout;

    /**
     * Create a new notification instance.
     */
    public function __construct(Payment $payment, Booking $booking, $commissionEarning, $payout)
    {
        $this->payment = $payment;
        $this->booking = $booking;
        $this->commissionEarning = $commissionEarning;
        $this->payout = $payout;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification for database.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'payment_id' => $this->payment->payment_id,
            'booking_id' => $this->booking->booking_id,
            'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
            'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
            'total_amount' => $this->payment->amount,
            'currency' => $this->payment->currency,
            'commission_rate' => $this->payment->commission_rate,
            'commission_amount' => $this->payment->commission_amount,
            'babysitter_payout_amount' => $this->payment->babysitter_payout_amount,
            'platform_earnings' => $this->payment->commission_amount,
            'care_type' => $this->booking->care_type,
            'start_date' => $this->booking->start_date->toDateString(),
            'end_date' => $this->booking->end_date ? $this->booking->end_date->toDateString() : null,
            'total_weeks' => $this->booking->total_weeks,
            'total_hours' => $this->booking->total_hours,
            'commission_earning_id' => $this->commissionEarning ? $this->commissionEarning->earning_id : null,
            'payout_id' => $this->payout ? $this->payout->payout_id : null,
            'payout_scheduled_date' => $this->payout ? $this->payout->scheduled_payout_date->toDateString() : null,
            'financial_summary' => [
                'parent_paid' => $this->payment->amount,
                'platform_commission' => $this->payment->commission_amount,
                'babysitter_earnings' => $this->payment->babysitter_payout_amount,
                'net_platform_earning' => $this->payment->commission_amount
            ],
            'message' => 'New payment received! Parent: ' . $this->booking->parent->first_name . ', Babysitter: ' . $this->booking->babysitter->first_name,
            'timestamp' => now()->toDateTimeString(),
            'icon' => '💰',
            'action_url' => '/admin/payments/' . $this->payment->payment_id
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'payment_id' => $this->payment->payment_id,
                'booking_id' => $this->booking->booking_id,
                'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
                'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
                'total_amount' => $this->payment->amount,
                'currency' => $this->payment->currency,
                'commission_rate' => $this->payment->commission_rate,
                'commission_amount' => $this->payment->commission_amount,
                'babysitter_payout_amount' => $this->payment->babysitter_payout_amount,
                'platform_earnings' => $this->payment->commission_amount,
                'care_type' => $this->booking->care_type,
                'start_date' => $this->booking->start_date->toDateString(),
                'end_date' => $this->booking->end_date ? $this->booking->end_date->toDateString() : null,
                'total_weeks' => $this->booking->total_weeks,
                'total_hours' => $this->booking->total_hours,
                'commission_earning_id' => $this->commissionEarning ? $this->commissionEarning->earning_id : null,
                'payout_id' => $this->payout ? $this->payout->payout_id : null,
                'payout_scheduled_date' => $this->payout ? $this->payout->scheduled_payout_date->toDateString() : null,
                'financial_summary' => [
                    'parent_paid' => $this->payment->amount,
                    'platform_commission' => $this->payment->commission_amount,
                    'babysitter_earnings' => $this->payment->babysitter_payout_amount,
                    'net_platform_earning' => $this->payment->commission_amount
                ],
                'message' => 'New payment received! Parent: ' . $this->booking->parent->first_name . ', Babysitter: ' . $this->booking->babysitter->first_name,
                'timestamp' => now()->toDateTimeString(),
                'icon' => '💰',
                'action_url' => '/admin/payments/' . $this->payment->payment_id
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'payment_id' => $this->payment->payment_id,
            'booking_id' => $this->booking->booking_id,
            'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
            'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
            'total_amount' => $this->payment->amount,
            'commission_amount' => $this->payment->commission_amount,
            'babysitter_payout_amount' => $this->payment->babysitter_payout_amount,
            'message' => 'New payment received with commission earnings.',
            'action_url' => '/admin/payments/' . $this->payment->payment_id
        ];
    }
}
