<?php

namespace App\Notifications;

use App\Models\Booking;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class SuperAdminBookingNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $booking;
    public $eventType;

    public function __construct(Booking $booking, string $eventType)
    {
        $this->booking = $booking;
        $this->eventType = $eventType;
    }

    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    public function toDatabase(object $notifiable): array
    {
        $messages = [
            'started' => "Booking #{$this->booking->booking_id} has started between {$this->booking->parent->first_name} and {$this->booking->babysitter->first_name}",
            'completed' => "Booking #{$this->booking->booking_id} has been completed between {$this->booking->parent->first_name} and {$this->booking->babysitter->first_name}"
        ];

        return [
            'booking_id' => $this->booking->booking_id,
            'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
            'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
            'care_type' => $this->booking->care_type,
            'total_amount' => $this->booking->total_amount,
            'commission_amount' => $this->booking->commission_amount,
            'event_type' => $this->eventType,
            'message' => $messages[$this->eventType] ?? "Booking status updated",
            'timestamp' => now()->toDateTimeString(),
            'icon' => $this->eventType === 'started' ? '🚀' : '✅',
            'action_url' => '/admin/bookings/' . $this->booking->booking_id
        ];
    }

    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        $messages = [
            'started' => "Booking #{$this->booking->booking_id} has started between {$this->booking->parent->first_name} and {$this->booking->babysitter->first_name}",
            'completed' => "Booking #{$this->booking->booking_id} has been completed between {$this->booking->parent->first_name} and {$this->booking->babysitter->first_name}"
        ];

        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'booking_id' => $this->booking->booking_id,
                'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
                'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
                'care_type' => $this->booking->care_type,
                'total_amount' => $this->booking->total_amount,
                'commission_amount' => $this->booking->commission_amount,
                'event_type' => $this->eventType,
                'message' => $messages[$this->eventType] ?? "Booking status updated",
                'timestamp' => now()->toDateTimeString(),
                'icon' => $this->eventType === 'started' ? '🚀' : '✅',
                'action_url' => '/admin/bookings/' . $this->booking->booking_id
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }
}
