<?php

namespace App\Notifications;

use App\Models\Payout;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class PayoutCreatedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $payout;

    /**
     * Create a new notification instance.
     */
    public function __construct(Payout $payout)
    {
        $this->payout = $payout;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification for database.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'payout_id' => $this->payout->payout_id,
            'amount' => $this->payout->amount,
            'currency' => $this->payout->currency,
            'payout_method' => $this->payout->payout_method,
            'payout_status' => $this->payout->payout_status,
            'scheduled_payout_date' => $this->payout->scheduled_payout_date->toDateString(),
            'estimated_processing_days' => 7,
            'booking_id' => $this->payout->payment->booking_id ?? null,
            'parent_name' => $this->payout->payment->booking->parent->first_name . ' ' . $this->payout->payment->booking->parent->last_name ?? 'Unknown',
            'care_type' => $this->payout->payment->booking->care_type ?? 'Unknown',
            'message' => 'Payout of ' . $this->payout->amount . ' ' . $this->payout->currency . ' scheduled for ' . $this->payout->scheduled_payout_date->format('M d, Y'),
            'timestamp' => now()->toDateTimeString(),
            'icon' => '💸',
            'action_url' => '/payouts/' . $this->payout->payout_id
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'payout_id' => $this->payout->payout_id,
                'amount' => $this->payout->amount,
                'currency' => $this->payout->currency,
                'payout_method' => $this->payout->payout_method,
                'payout_status' => $this->payout->payout_status,
                'scheduled_payout_date' => $this->payout->scheduled_payout_date->toDateString(),
                'estimated_processing_days' => 7,
                'booking_id' => $this->payout->payment->booking_id ?? null,
                'parent_name' => $this->payout->payment->booking->parent->first_name . ' ' . $this->payout->payment->booking->parent->last_name ?? 'Unknown',
                'care_type' => $this->payout->payment->booking->care_type ?? 'Unknown',
                'message' => 'Payout of ' . $this->payout->amount . ' ' . $this->payout->currency . ' scheduled for ' . $this->payout->scheduled_payout_date->format('M d, Y'),
                'timestamp' => now()->toDateTimeString(),
                'icon' => '💸',
                'action_url' => '/payouts/' . $this->payout->payout_id
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'payout_id' => $this->payout->payout_id,
            'amount' => $this->payout->amount,
            'currency' => $this->payout->currency,
            'payout_method' => $this->payout->payout_method,
            'scheduled_payout_date' => $this->payout->scheduled_payout_date->toDateString(),
            'message' => 'Payout scheduled for your earnings.',
            'action_url' => '/payouts/' . $this->payout->payout_id
        ];
    }
}
