<?php

namespace App\Notifications;

use App\Models\Payment;
use App\Models\Booking;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class PaymentSuccessNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $payment;
    public $booking;

    /**
     * Create a new notification instance.
     */
    public function __construct(Payment $payment, Booking $booking)
    {
        $this->payment = $payment;
        $this->booking = $booking;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification for database.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'payment_id' => $this->payment->payment_id,
            'booking_id' => $this->booking->booking_id,
            'amount' => $this->payment->amount,
            'currency' => $this->payment->currency,
            'transaction_id' => $this->payment->razorpay_payment_id,
            'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
            'care_type' => $this->booking->care_type,
            'start_date' => $this->booking->start_date->toDateString(),
            'end_date' => $this->booking->end_date ? $this->booking->end_date->toDateString() : null,
            'total_weeks' => $this->booking->total_weeks,
            'total_hours' => $this->booking->total_hours,
            'message' => 'Payment successful! Your booking with ' . $this->booking->babysitter->first_name . ' has been confirmed.',
            'timestamp' => now()->toDateTimeString(),
            'icon' => '💰',
            'action_url' => '/bookings/' . $this->booking->booking_id
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'payment_id' => $this->payment->payment_id,
                'booking_id' => $this->booking->booking_id,
                'amount' => $this->payment->amount,
                'currency' => $this->payment->currency,
                'transaction_id' => $this->payment->razorpay_payment_id,
                'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
                'care_type' => $this->booking->care_type,
                'start_date' => $this->booking->start_date->toDateString(),
                'end_date' => $this->booking->end_date ? $this->booking->end_date->toDateString() : null,
                'total_weeks' => $this->booking->total_weeks,
                'total_hours' => $this->booking->total_hours,
                'message' => 'Payment successful! Your booking with ' . $this->booking->babysitter->first_name . ' has been confirmed.',
                'timestamp' => now()->toDateTimeString(),
                'icon' => '💰',
                'action_url' => '/bookings/' . $this->booking->booking_id
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'payment_id' => $this->payment->payment_id,
            'booking_id' => $this->booking->booking_id,
            'amount' => $this->payment->amount,
            'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
            'message' => 'Payment successful! Booking confirmed.',
            'action_url' => '/bookings/' . $this->booking->booking_id
        ];
    }
}
