<?php

namespace App\Notifications;

use App\Models\ParentProfile;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class ParentProfileNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $profile;
    public $action;

    /**
     * Create a new notification instance.
     */
    public function __construct(ParentProfile $profile, string $action)
    {
        $this->profile = $profile;
        $this->action = $action;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification for database.
     */
    public function toDatabase(object $notifiable): array
    {
        // Role based user type determine karein
        $userType = 'User';
        if ($this->profile->user->role_id == 2) {
            $userType = 'Parent';
        } elseif ($this->profile->user->role_id == 3) {
            $userType = 'Babysitter';
        }

        return [
            'profile_id' => $this->profile->profile_id,
            'user_id' => $this->profile->user_id,
            'user_name' => $this->profile->user->first_name . ' ' . $this->profile->user->last_name,
            'user_email' => $this->profile->user->email,
            'user_type' => $userType,
            'role_id' => $this->profile->user->role_id,
            'action' => $this->action,
            'message' => 'Parent profile ' . $this->action . ': ' . $this->profile->user->first_name . ' ' . $this->profile->user->last_name,
            'profile_picture' => $this->profile->user->profile_picture_url,
            'family_description' => $this->profile->family_description,
            'preferred_communication' => $this->profile->preferred_communication,
            'care_recipients_count' => $this->profile->careRecipients->count(),
            'timestamp' => now()->toDateTimeString(),
            'action_url' => '/admin/parents/' . $this->profile->profile_id,
            'icon' => '👨‍👩‍👧‍👦'
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        // Role based user type determine karein
        $userType = 'User';
        if ($this->profile->user->role_id == 2) {
            $userType = 'Parent';
        } elseif ($this->profile->user->role_id == 3) {
            $userType = 'Babysitter';
        }

        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'profile_id' => $this->profile->profile_id,
                'user_id' => $this->profile->user_id,
                'user_name' => $this->profile->user->first_name . ' ' . $this->profile->user->last_name,
                'user_email' => $this->profile->user->email,
                'user_type' => $userType,
                'role_id' => $this->profile->user->role_id,
                'action' => $this->action,
                'message' => 'Parent profile ' . $this->action . ': ' . $this->profile->user->first_name . ' ' . $this->profile->user->last_name,
                'profile_picture' => $this->profile->user->profile_picture_url,
                'family_description' => $this->profile->family_description,
                'preferred_communication' => $this->profile->preferred_communication,
                'care_recipients_count' => $this->profile->careRecipients->count(),
                'timestamp' => now()->toDateTimeString(),
                'action_url' => '/admin/parents/' . $this->profile->profile_id,
                'icon' => '👨‍👩‍👧‍👦'
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'profile_id' => $this->profile->profile_id,
            'user_id' => $this->profile->user_id,
            'user_name' => $this->profile->user->first_name . ' ' . $this->profile->user->last_name,
            'user_email' => $this->profile->user->email,
            'action' => $this->action,
            'message' => 'Parent profile ' . $this->action,
            'action_url' => '/admin/parents/' . $this->profile->profile_id
        ];
    }
}
