<?php

namespace App\Notifications;

use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class NewUserRegistrationNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $user;
    public $userType;

    /**
     * Create a new notification instance.
     */
    public function __construct(User $user, string $userType)
    {
        $this->user = $user;
        $this->userType = $userType;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        // ✅ MAIL CHANNEL REMOVE KAR DIYA - sirf database aur broadcast
        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification for database.
     *
     * @return array<string, mixed>
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'user_id' => $this->user->user_id,
            'user_name' => $this->user->first_name . ' ' . $this->user->last_name,
            'user_email' => $this->user->email,
            'user_type' => $this->userType,
            'message' => 'New ' . $this->userType . ' registered: ' . $this->user->first_name . ' ' . $this->user->last_name,
            'profile_picture' => $this->user->profile_picture_url,
            'registration_date' => $this->user->created_at->toDateTimeString(),
            'action_url' => '/admin/users/' . $this->user->user_id,
            'icon' => '👤'
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'user_id' => $this->user->user_id,
                'user_name' => $this->user->first_name . ' ' . $this->user->last_name,
                'user_email' => $this->user->email,
                'user_type' => $this->userType,
                'message' => 'New ' . $this->userType . ' registered: ' . $this->user->first_name . ' ' . $this->user->last_name,
                'profile_picture' => $this->user->profile_picture_url,
                'registration_date' => $this->user->created_at->toDateTimeString(),
                'action_url' => '/admin/users/' . $this->user->user_id,
                'icon' => '👤',
                'timestamp' => now()->toDateTimeString()
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'user_id' => $this->user->user_id,
            'user_name' => $this->user->first_name . ' ' . $this->user->last_name,
            'user_email' => $this->user->email,
            'user_type' => $this->userType,
            'message' => 'New ' . $this->userType . ' registered: ' . $this->user->first_name . ' ' . $this->user->last_name,
            'action_url' => '/admin/users/' . $this->user->user_id
        ];
    }
}
