<?php

namespace App\Notifications;

use App\Models\DirectBookingRequest;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class BookingRejectedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $bookingRequest;

    /**
     * Create a new notification instance.
     */
    public function __construct(DirectBookingRequest $bookingRequest)
    {
        $this->bookingRequest = $bookingRequest;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification for database.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'booking_request_id' => $this->bookingRequest->booking_request_id,
            'parent_id' => $this->bookingRequest->parent_id,
            'babysitter_id' => $this->bookingRequest->babysitter_id,
            'babysitter_name' => $this->bookingRequest->babysitter->first_name . ' ' . $this->bookingRequest->babysitter->last_name,
            'babysitter_email' => $this->bookingRequest->babysitter->email,
            'start_date' => $this->bookingRequest->start_date->toDateString(),
            'end_date' => $this->bookingRequest->end_date ? $this->bookingRequest->end_date->toDateString() : null,
            'expected_hours_per_week' => $this->bookingRequest->expected_hours_per_week,
            'hourly_rate' => $this->bookingRequest->hourly_rate,
            'total_expected_amount' => $this->bookingRequest->total_expected_amount,
            'care_type' => $this->bookingRequest->care_type,
            'request_status' => $this->bookingRequest->request_status,
            'payment_status' => $this->bookingRequest->payment_status,
            'rejected_at' => $this->bookingRequest->rejected_at,
            'message' => $this->bookingRequest->babysitter->first_name . ' has declined your booking request.',
            'timestamp' => now()->toDateTimeString(),
            'icon' => '❌',
            'action_url' => '/booking-requests/' . $this->bookingRequest->booking_request_id
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'booking_request_id' => $this->bookingRequest->booking_request_id,
                'parent_id' => $this->bookingRequest->parent_id,
                'babysitter_id' => $this->bookingRequest->babysitter_id,
                'babysitter_name' => $this->bookingRequest->babysitter->first_name . ' ' . $this->bookingRequest->babysitter->last_name,
                'babysitter_email' => $this->bookingRequest->babysitter->email,
                'start_date' => $this->bookingRequest->start_date->toDateString(),
                'end_date' => $this->bookingRequest->end_date ? $this->bookingRequest->end_date->toDateString() : null,
                'expected_hours_per_week' => $this->bookingRequest->expected_hours_per_week,
                'hourly_rate' => $this->bookingRequest->hourly_rate,
                'total_expected_amount' => $this->bookingRequest->total_expected_amount,
                'care_type' => $this->bookingRequest->care_type,
                'request_status' => $this->bookingRequest->request_status,
                'payment_status' => $this->bookingRequest->payment_status,
                'rejected_at' => $this->bookingRequest->rejected_at,
                'message' => $this->bookingRequest->babysitter->first_name . ' has declined your booking request.',
                'timestamp' => now()->toDateTimeString(),
                'icon' => '❌',
                'action_url' => '/booking-requests/' . $this->bookingRequest->booking_request_id
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'booking_request_id' => $this->bookingRequest->booking_request_id,
            'parent_id' => $this->bookingRequest->parent_id,
            'babysitter_id' => $this->bookingRequest->babysitter_id,
            'babysitter_name' => $this->bookingRequest->babysitter->first_name . ' ' . $this->bookingRequest->babysitter->last_name,
            'message' => 'Your booking request has been declined.',
            'action_url' => '/booking-requests/' . $this->bookingRequest->booking_request_id
        ];
    }
}
