<?php

namespace App\Notifications;

use App\Models\Booking;
use App\Models\Payment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class BookingPaymentReceivedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $booking;
    public $payment;

    /**
     * Create a new notification instance.
     */
    public function __construct(Booking $booking, Payment $payment)
    {
        $this->booking = $booking;
        $this->payment = $payment;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification for database.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'booking_id' => $this->booking->booking_id,
            'payment_id' => $this->payment->payment_id,
            'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
            'amount' => $this->payment->amount,
            'currency' => $this->payment->currency,
            'babysitter_earnings' => $this->booking->babysitter_earnings,
            'care_type' => $this->booking->care_type,
            'start_date' => $this->booking->start_date->toDateString(),
            'end_date' => $this->booking->end_date ? $this->booking->end_date->toDateString() : null,
            'total_weeks' => $this->booking->total_weeks,
            'total_hours' => $this->booking->total_hours,
            'payout_scheduled_date' => now()->addDays(7)->toDateString(),
            'message' => 'Payment received from ' . $this->booking->parent->first_name . ' for your booking! Payout scheduled.',
            'timestamp' => now()->toDateTimeString(),
            'icon' => '💳',
            'action_url' => '/bookings/' . $this->booking->booking_id
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'booking_id' => $this->booking->booking_id,
                'payment_id' => $this->payment->payment_id,
                'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
                'amount' => $this->payment->amount,
                'currency' => $this->payment->currency,
                'babysitter_earnings' => $this->booking->babysitter_earnings,
                'care_type' => $this->booking->care_type,
                'start_date' => $this->booking->start_date->toDateString(),
                'end_date' => $this->booking->end_date ? $this->booking->end_date->toDateString() : null,
                'total_weeks' => $this->booking->total_weeks,
                'total_hours' => $this->booking->total_hours,
                'payout_scheduled_date' => now()->addDays(7)->toDateString(),
                'message' => 'Payment received from ' . $this->booking->parent->first_name . ' for your booking! Payout scheduled.',
                'timestamp' => now()->toDateTimeString(),
                'icon' => '💳',
                'action_url' => '/bookings/' . $this->booking->booking_id
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'booking_id' => $this->booking->booking_id,
            'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
            'amount' => $this->payment->amount,
            'babysitter_earnings' => $this->booking->babysitter_earnings,
            'message' => 'Payment received! Payout scheduled.',
            'action_url' => '/bookings/' . $this->booking->booking_id
        ];
    }
}
