<?php

namespace App\Notifications;

use App\Models\Booking;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\BroadcastMessage;

class BookingCompletedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $booking;
    public $recipientType;

    public function __construct(Booking $booking, string $recipientType)
    {
        $this->booking = $booking;
        $this->recipientType = $recipientType;
    }

    public function via(object $notifiable): array
    {
        return ['database', 'broadcast'];
    }

    public function toDatabase(object $notifiable): array
    {
        $message = $this->recipientType === 'parent'
            ? "Your booking with {$this->booking->babysitter->first_name} has been completed successfully!"
            : "You have successfully completed the booking with {$this->booking->parent->first_name}!";

        return [
            'booking_id' => $this->booking->booking_id,
            'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
            'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
            'care_type' => $this->booking->care_type,
            'start_date' => $this->booking->start_date->toDateString(),
            'completed_at' => $this->booking->completed_at?->toDateTimeString(),
            'total_weeks' => $this->booking->total_weeks,
            'total_hours' => $this->booking->total_hours,
            'message' => $message,
            'timestamp' => now()->toDateTimeString(),
            'icon' => '✅',
            'action_url' => '/bookings/' . $this->booking->booking_id
        ];
    }

    public function toBroadcast(object $notifiable): BroadcastMessage
    {
        $message = $this->recipientType === 'parent'
            ? "Your booking with {$this->booking->babysitter->first_name} has been completed successfully!"
            : "You have successfully completed the booking with {$this->booking->parent->first_name}!";

        return new BroadcastMessage([
            'id' => $this->id,
            'type' => __CLASS__,
            'data' => [
                'booking_id' => $this->booking->booking_id,
                'parent_name' => $this->booking->parent->first_name . ' ' . $this->booking->parent->last_name,
                'babysitter_name' => $this->booking->babysitter->first_name . ' ' . $this->booking->babysitter->last_name,
                'care_type' => $this->booking->care_type,
                'start_date' => $this->booking->start_date->toDateString(),
                'completed_at' => $this->booking->completed_at?->toDateTimeString(),
                'total_weeks' => $this->booking->total_weeks,
                'total_hours' => $this->booking->total_hours,
                'message' => $message,
                'timestamp' => now()->toDateTimeString(),
                'icon' => '✅',
                'action_url' => '/bookings/' . $this->booking->booking_id
            ],
            'read_at' => null,
            'created_at' => now()->toDateTimeString(),
        ]);
    }
}
