<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Review extends Model
{
    use HasFactory;

    protected $primaryKey = 'review_id';

    protected $fillable = [
        'booking_id',
        'reviewer_id',
        'reviewed_user_id',
        'review_type',
        'rating',
        'review_text',
        'would_recommend',
        'strengths',
        'areas_for_improvement',
        'is_public',
        'is_verified_booking',
        'helpful_count',
        'reported_count'
    ];

    protected $casts = [
        'rating' => 'integer',
        'would_recommend' => 'boolean',
        'is_public' => 'boolean',
        'is_verified_booking' => 'boolean',
        'helpful_count' => 'integer',
        'reported_count' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get the booking associated with the review.
     */
    public function booking()
    {
        return $this->belongsTo(Booking::class, 'booking_id', 'booking_id');
    }

    /**
     * Get the user who wrote the review.
     */
    public function reviewer()
    {
        return $this->belongsTo(User::class, 'reviewer_id', 'user_id');
    }

    /**
     * Get the user who is being reviewed.
     */
    public function reviewedUser()
    {
        return $this->belongsTo(User::class, 'reviewed_user_id', 'user_id');
    }

    /**
     * Scope a query to only include public reviews.
     */
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    /**
     * Scope a query to only include reviews for a specific user.
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('reviewed_user_id', $userId);
    }

    /**
     * Scope a query to only include parent to babysitter reviews.
     */
    public function scopeParentToBabysitter($query)
    {
        return $query->where('review_type', 'parent_to_babysitter');
    }

    /**
     * Scope a query to only include babysitter to parent reviews.
     */
    public function scopeBabysitterToParent($query)
    {
        return $query->where('review_type', 'babysitter_to_parent');
    }

    /**
     * Check if review is from parent to babysitter.
     */
    public function isParentToBabysitter(): bool
    {
        return $this->review_type === 'parent_to_babysitter';
    }

    /**
     * Check if review is from babysitter to parent.
     */
    public function isBabysitterToParent(): bool
    {
        return $this->review_type === 'babysitter_to_parent';
    }

    /**
     * Mark review as helpful.
     */
    public function markAsHelpful()
    {
        $this->increment('helpful_count');
    }

    /**
     * Report review.
     */
    public function report()
    {
        $this->increment('reported_count');
    }

    /**
     * Hide review from public.
     */
    public function hideFromPublic()
    {
        $this->update(['is_public' => false]);
    }

    /**
     * Show review to public.
     */
    public function showToPublic()
    {
        $this->update(['is_public' => true]);
    }
}
