<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ReportTemplate extends Model
{
    use HasFactory;

    protected $primaryKey = 'template_id';

    protected $fillable = [
        'template_name',
        'template_type',
        'template_query',
        'parameters',
        'is_active',
        'created_by'
    ];

    protected $casts = [
        'parameters' => 'array', // Yeh automatically JSON string ko array mein convert karega
        'is_active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get the user who created the template.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by', 'user_id');
    }

    /**
     * Scope a query to only include active templates.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include templates of specific type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('template_type', $type);
    }

    /**
     * Check if template is active.
     */
    public function isActive(): bool
    {
        return $this->is_active;
    }

    /**
     * Activate the template.
     */
    public function activate()
    {
        $this->update(['is_active' => true]);
    }

    /**
     * Deactivate the template.
     */
    public function deactivate()
    {
        $this->update(['is_active' => false]);
    }

    /**
     * Get template parameters.
     */
    public function getParameters()
    {
        // Cast automatically handle karega, but ensure array return
        return $this->parameters ?? [];
    }

    /**
     * Set template parameters.
     */
    public function setParameters(array $parameters)
    {
        $this->update(['parameters' => $parameters]);
    }

    /**
     * Check if template has parameters.
     */
    public function hasParameters(): bool
    {
        return !empty($this->parameters);
    }
}
