<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Payout extends Model
{
    use HasFactory;

    protected $primaryKey = 'payout_id';

    protected $fillable = [
        'babysitter_id',
        'payment_id',
        'amount',
         'currency', // ✅ ADD THIS LINE
        'payout_method',
        'payout_status',
        'payout_reference',
        'payout_notes',
        'scheduled_payout_date',
        'processed_at',
        'completed_at',
        'failed_at'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'scheduled_payout_date' => 'date',
        'processed_at' => 'datetime',
        'completed_at' => 'datetime',
        'failed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get the babysitter associated with the payout.
     */
    public function babysitter()
    {
        return $this->belongsTo(User::class, 'babysitter_id', 'user_id');
    }

    /**
     * Get the payment associated with the payout.
     */
    public function payment()
    {
        return $this->belongsTo(Payment::class, 'payment_id', 'payment_id');
    }

    /**
     * Scope a query to only include pending payouts.
     */
    public function scopePending($query)
    {
        return $query->where('payout_status', 'pending');
    }

    /**
     * Scope a query to only include completed payouts.
     */
    public function scopeCompleted($query)
    {
        return $query->where('payout_status', 'completed');
    }

    /**
     * Scope a query to only include payouts for a specific babysitter.
     */
    public function scopeForBabysitter($query, $babysitterId)
    {
        return $query->where('babysitter_id', $babysitterId);
    }

    /**
     * Check if payout is pending.
     */
    public function isPending(): bool
    {
        return $this->payout_status === 'pending';
    }

    /**
     * Check if payout is completed.
     */
    public function isCompleted(): bool
    {
        return $this->payout_status === 'completed';
    }

    /**
     * Check if payout is processing.
     */
    public function isProcessing(): bool
    {
        return $this->payout_status === 'processing';
    }

    /**
     * Mark payout as processing.
     */
    public function markAsProcessing()
    {
        $this->update([
            'payout_status' => 'processing',
            'processed_at' => now()
        ]);
    }

    /**
     * Mark payout as completed.
     */
    public function markAsCompleted($payoutReference = null)
    {
        $this->update([
            'payout_status' => 'completed',
            'payout_reference' => $payoutReference,
            'completed_at' => now()
        ]);
    }

    /**
     * Mark payout as failed.
     */
    public function markAsFailed($notes = null)
    {
        $this->update([
            'payout_status' => 'failed',
            'payout_notes' => $notes,
            'failed_at' => now()
        ]);
    }
}
