<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class Payment extends Model
{
    use HasFactory;

    protected $primaryKey = 'payment_id';

    protected $fillable = [
        'booking_id',
        'parent_id',
        'babysitter_id',
        'amount',
        'currency',
        'exchange_rate',
        'base_currency_amount',
        'subtotal_amount',
        'commission_rate',
        'commission_amount',
        'babysitter_payout_amount',
        'payment_method',
        'payment_gateway',
        'razorpay_order_id',
        'razorpay_payment_id',
        'razorpay_signature',
        'transaction_id',
        'gateway_response',
        'payment_status',
        'paid_at',
        'processed_at',
        'payout_sent_at',
        'refunded_at'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'base_currency_amount' => 'decimal:2',
        'subtotal_amount' => 'decimal:2',
        'commission_rate' => 'decimal:2',
        'commission_amount' => 'decimal:2',
        'babysitter_payout_amount' => 'decimal:2',
        'exchange_rate' => 'decimal:6',
        'paid_at' => 'datetime',
        'processed_at' => 'datetime',
        'payout_sent_at' => 'datetime',
        'refunded_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // ✅ ENHANCED: Status Check Methods
    public function canAcceptPayment(): bool
    {
        return in_array($this->payment_status, ['pending', 'failed']);
    }
    public function payouts()
    {
        return $this->hasMany(Payout::class, 'payment_id', 'payment_id');
    }

    public function canRefund(): bool
    {
        return $this->payment_status === 'completed' &&
            !$this->refunded_at &&
            $this->paid_at &&
            $this->paid_at->gt(now()->subDays(90)); // Within 90 days
    }

    public function isProcessable(): bool
    {
        return in_array($this->payment_status, ['pending', 'failed']);
    }

    public function isRefundable(): bool
    {
        return $this->payment_status === 'completed' &&
            is_null($this->refunded_at) &&
            $this->paid_at;
    }

    public function getRefundableAmount(): float
    {
        if (!$this->isRefundable()) {
            return 0;
        }

        // If already partially refunded, calculate remaining amount
        $refundedAmount = $this->refunded_amount ?? 0;
        return max(0, $this->amount - $refundedAmount);
    }

    // ✅ NEW: Scope for pending and failed payments
    public function scopePendingOrFailed($query)
    {
        return $query->whereIn('payment_status', ['pending', 'failed']);
    }

    public function scopeRefundable($query)
    {
        return $query->where('payment_status', 'completed')
            ->whereNull('refunded_at')
            ->whereNotNull('paid_at')
            ->where('paid_at', '>', now()->subDays(90));
    }

    public function scopeExpired($query)
    {
        return $query->where('payment_status', 'pending')
            ->where('created_at', '<', now()->subHours(24));
    }

    // Relationships
    public function booking()
    {
        return $this->belongsTo(Booking::class, 'booking_id', 'booking_id');
    }

    public function parent()
    {
        return $this->belongsTo(User::class, 'parent_id', 'user_id');
    }

    public function babysitter()
    {
        return $this->belongsTo(User::class, 'babysitter_id', 'user_id');
    }

    // Scopes
    public function scopeCompleted($query)
    {
        return $query->where('payment_status', 'completed');
    }

    public function scopePending($query)
    {
        return $query->where('payment_status', 'pending');
    }

    public function scopeForParent($query, $parentId)
    {
        return $query->where('parent_id', $parentId);
    }

    public function scopeForBabysitter($query, $babysitterId)
    {
        return $query->where('babysitter_id', $babysitterId);
    }

    public function scopeRazorpayOrders($query)
    {
        return $query->whereNotNull('razorpay_order_id');
    }

    // Methods
    public function isCompleted(): bool
    {
        return $this->payment_status === 'completed';
    }

    public function isPending(): bool
    {
        return $this->payment_status === 'pending';
    }

    public function isRefunded(): bool
    {
        return $this->payment_status === 'refunded';
    }

    public function isRazorpay(): bool
    {
        return $this->payment_gateway === 'razorpay';
    }

    public function markAsProcessing()
    {
        if (!$this->canAcceptPayment()) {
            throw new \Exception('Payment cannot be processed in current status: ' . $this->payment_status);
        }

        $this->update([
            'payment_status' => 'processing',
            'processed_at' => now()
        ]);

        Log::info('✅ PAYMENT PROCESSING - Marked as processing:', [
            'payment_id' => $this->payment_id,
            'booking_id' => $this->booking_id
        ]);
    }

    public function markAsCompleted($razorpayPaymentId = null, $razorpaySignature = null, $gatewayResponse = null)
    {
        if (!$this->canAcceptPayment()) {
            throw new \Exception('Payment cannot be completed in current status: ' . $this->payment_status);
        }

        $updateData = [
            'payment_status' => 'completed',
            'razorpay_payment_id' => $razorpayPaymentId,
            'razorpay_signature' => $razorpaySignature,
            'gateway_response' => $gatewayResponse,
            'paid_at' => now()
        ];

        $this->update($updateData);

        Log::info('✅ PAYMENT COMPLETED - Successfully marked:', [
            'payment_id' => $this->payment_id,
            'booking_id' => $this->booking_id,
            'razorpay_payment_id' => $razorpayPaymentId
        ]);

        // Update booking payment status
        if ($this->booking && $this->booking->canConfirmAfterPayment()) {
            $this->booking->markPaymentAsPaid();
        }
    }

    public function markAsFailed($gatewayResponse = null)
    {
        $this->update([
            'payment_status' => 'failed',
            'gateway_response' => $gatewayResponse
        ]);

        Log::warning('⚠️ PAYMENT FAILED - Marked as failed:', [
            'payment_id' => $this->payment_id,
            'gateway_response' => $gatewayResponse
        ]);
    }

    public function markAsRefunded($refundAmount = null, $refundId = null)
    {
        if (!$this->canRefund()) {
            throw new \Exception('Payment cannot be refunded in current status');
        }

        $refundAmount = $refundAmount ?? $this->amount;

        $this->update([
            'payment_status' => 'refunded',
            'refunded_at' => now(),
            'refunded_amount' => $refundAmount,
            'refund_id' => $refundId
        ]);

        Log::info('💰 PAYMENT REFUNDED - Successfully processed:', [
            'payment_id' => $this->payment_id,
            'refund_amount' => $refundAmount,
            'refund_id' => $refundId
        ]);

        // Update booking status if needed
        if ($this->booking) {
            $this->booking->update(['payment_status' => 'refunded']);
        }
    }

    public function markAsExpired()
    {
        if ($this->payment_status !== 'pending') {
            throw new \Exception('Only pending payments can be marked as expired');
        }

        $this->update([
            'payment_status' => 'expired',
            'gateway_response' => 'Payment expired after 24 hours'
        ]);

        Log::info('⏰ PAYMENT EXPIRED - Marked as expired:', [
            'payment_id' => $this->payment_id,
            'booking_id' => $this->booking_id
        ]);
    }

    public function getAmountInPaise()
    {
        return (int) ($this->amount * 100); // Razorpay expects amount in paise
    }

    public function verifyRazorpaySignature($attributes)
    {
        $generatedSignature = hash_hmac('sha256', $attributes['razorpay_order_id'] . '|' . $attributes['razorpay_payment_id'], env('RAZORPAY_SECRET'));

        return hash_equals($generatedSignature, $attributes['razorpay_signature']);
    }

    // ✅ NEW: Get payment statistics
    public function getPaymentDuration()
    {
        if (!$this->paid_at || !$this->created_at) {
            return null;
        }

        return $this->created_at->diffInMinutes($this->paid_at);
    }

    // ✅ NEW: Check if payment is stuck
    public function isStuck()
    {
        return $this->payment_status === 'processing' &&
            $this->processed_at &&
            $this->processed_at->lt(now()->subHours(2));
    }
}
