<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class LanguageTranslation extends Model
{
    use HasFactory;

    protected $primaryKey = 'translation_id';

    protected $fillable = [
        'language_code',
        'translation_key',
        'translation_value',
        'page_section'
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get the language that owns the translation.
     */
    public function language()
    {
        return $this->belongsTo(Language::class, 'language_code', 'language_code');
    }

    /**
     * Scope translations by language code
     */
    public function scopeByLanguage($query, $languageCode)
    {
        return $query->where('language_code', $languageCode);
    }

    /**
     * Scope translations by page section
     */
    public function scopeBySection($query, $section)
    {
        return $query->where('page_section', $section);
    }

    /**
     * Scope translations by key
     */
    public function scopeByKey($query, $key)
    {
        return $query->where('translation_key', $key);
    }

    /**
     * Get translation value by key and language
     */
    public static function getTranslation($languageCode, $key, $default = null)
    {
        $translation = self::where('language_code', $languageCode)
            ->where('translation_key', $key)
            ->first();

        return $translation ? $translation->translation_value : $default;
    }

    /**
     * Get all translations for a language as key-value array
     */
    public static function getAllTranslations($languageCode)
    {
        return self::where('language_code', $languageCode)
            ->pluck('translation_value', 'translation_key')
            ->toArray();
    }

    /**
     * Get translations grouped by section for a language
     */
    public static function getTranslationsBySection($languageCode)
    {
        return self::where('language_code', $languageCode)
            ->get()
            ->groupBy('page_section')
            ->map(function ($items) {
                return $items->pluck('translation_value', 'translation_key')->toArray();
            })
            ->toArray();
    }
}
