<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BookingType extends Model
{
    use HasFactory;

    protected $primaryKey = 'type_id';

    protected $fillable = [
        'type_name',
        'type_description',
        'min_duration_days',
        'max_duration_days',
        'requires_contract',
        'is_active'
    ];

    protected $casts = [
        'min_duration_days' => 'integer',
        'max_duration_days' => 'integer',
        'requires_contract' => 'boolean',
        'is_active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Scope active booking types
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope by duration range
     */
    public function scopeByDuration($query, $minDays = null, $maxDays = null)
    {
        if ($minDays) {
            $query->where('min_duration_days', '<=', $minDays);
        }

        if ($maxDays) {
            $query->where(function($q) use ($maxDays) {
                $q->whereNull('max_duration_days')
                  ->orWhere('max_duration_days', '>=', $maxDays);
            });
        }

        return $query;
    }

    /**
     * Scope that require contract
     */
    public function scopeRequiresContract($query)
    {
        return $query->where('requires_contract', true);
    }

    /**
     * Check if duration is valid for this booking type
     */
    public function isValidDuration($durationDays)
    {
        if ($durationDays < $this->min_duration_days) {
            return false;
        }

        if ($this->max_duration_days && $durationDays > $this->max_duration_days) {
            return false;
        }

        return true;
    }

    /**
     * Get booking type by name
     */
    public static function getByName($typeName)
    {
        return self::where('type_name', $typeName)->active()->first();
    }

    /**
     * Get available booking types for duration
     */
    public static function getAvailableForDuration($durationDays)
    {
        return self::active()
            ->where('min_duration_days', '<=', $durationDays)
            ->where(function($query) use ($durationDays) {
                $query->whereNull('max_duration_days')
                      ->orWhere('max_duration_days', '>=', $durationDays);
            })
            ->get();
    }
}
