<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BabysitterProfile extends Model
{
    use HasFactory;

    protected $primaryKey = 'profile_id';

    protected $fillable = [
        'user_id',
        'occupation_category_id',
        'hourly_rate',
        'hourly_rates',
        'experience_years',
        'availability_type',
        'immediate_availability',
        'willing_to_travel',
        'has_vehicle',
        'has_drivers_license',
        'about_me',
        'why_babysit',
        'is_profile_public',
        'is_verified',
        'verification_status',
        'total_reviews',
        'average_rating',
        'completed_jobs',
        'response_rate',
        'response_time_minutes',
        'last_search_update'
    ];

    protected $casts = [
        'hourly_rate' => 'decimal:2',
        'hourly_rates' => 'array',
        'experience_years' => 'integer',
        'immediate_availability' => 'boolean',
        'willing_to_travel' => 'boolean',
        'has_vehicle' => 'boolean',
        'has_drivers_license' => 'boolean',
        'is_profile_public' => 'boolean',
        'is_verified' => 'boolean',
        'total_reviews' => 'integer',
        'average_rating' => 'decimal:2',
        'completed_jobs' => 'integer',
        'response_rate' => 'decimal:2',
        'response_time_minutes' => 'integer',
        'last_search_update' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get the user that owns the babysitter profile.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'user_id');
    }

    /**
     * Get the occupation category that owns the babysitter profile.
     */
    public function occupationCategory()
    {
        return $this->belongsTo(OccupationCategory::class, 'occupation_category_id', 'category_id');
    }

    /**
     * Get the skills for the babysitter.
     */
    public function skills()
    {
        return $this->hasMany(BabysitterSkill::class, 'babysitter_id', 'profile_id');
    }

    /**
     * Scope public profiles
     */
    public function scopePublic($query)
    {
        return $query->where('is_profile_public', true);
    }

    /**
     * Scope verified profiles
     */
    public function scopeVerified($query)
    {
        return $query->where('is_verified', true);
    }

    /**
     * Scope active profiles
     */
    public function scopeActive($query)
    {
        return $query->whereHas('user', function($q) {
            $q->where('is_active', true);
        });
    }

    /**
     * Scope by availability type
     */
    public function scopeByAvailability($query, $availability)
    {
        return $query->where('availability_type', $availability);
    }

    /**
     * Scope by category type
     */
    public function scopeByCategoryType($query, $categoryType)
    {
        return $query->whereHas('occupationCategory', function($q) use ($categoryType) {
            $q->where('category_type', $categoryType);
        });
    }

    /**
     * Update search metrics
     */
    public function updateSearchMetrics()
    {
        $this->update([
            'last_search_update' => now()
        ]);
    }

    /**
 * Get the addresses for the babysitter through user.
 */
public function addresses()
{
    return $this->hasManyThrough(
        Address::class,
        User::class,
        'user_id', // Foreign key on users table
        'user_id', // Foreign key on addresses table
        'user_id', // Local key on babysitter_profiles table
        'user_id'  // Local key on users table
    );
}

/**
 * Get the primary address for the babysitter.
 */
public function primaryAddress()
{
    return $this->hasOneThrough(
        Address::class,
        User::class,
        'user_id', // Foreign key on users table
        'user_id', // Foreign key on addresses table
        'user_id', // Local key on babysitter_profiles table
        'user_id'  // Local key on users table
    )->where('is_primary', true);
}
}
