<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{

    public function index(Request $request): JsonResponse
    {
        try {
            $query = User::with(['role', 'addresses', 'userLanguages']);

            // ✅ Search filter
            if ($request->has('search') && !empty($request->search)) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('first_name', 'LIKE', "%{$search}%")
                        ->orWhere('last_name', 'LIKE', "%{$search}%")
                        ->orWhere('email', 'LIKE', "%{$search}%")
                        ->orWhere('phone_number', 'LIKE', "%{$search}%");
                });
            }

            // ✅ Role filter
            if ($request->has('role_id') && !empty($request->role_id)) {
                $query->where('role_id', $request->role_id);
            }

            // ✅ Status filter
            if ($request->has('status') && !empty($request->status)) {
                $query->where('status', $request->status);
            }

            // ✅ Gender filter
            if ($request->has('gender') && !empty($request->gender)) {
                $query->where('gender', $request->gender);
            }

            // ✅ Date range filter
            if ($request->has('date_from') && !empty($request->date_from)) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }

            if ($request->has('date_to') && !empty($request->date_to)) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            // ✅ Sort by
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');

            $allowedSortColumns = ['created_at', 'first_name', 'last_name', 'email', 'updated_at'];
            if (in_array($sortBy, $allowedSortColumns)) {
                $query->orderBy($sortBy, $sortOrder);
            } else {
                $query->orderBy('created_at', 'desc');
            }

            // ✅ Pagination
            $perPage = $request->get('per_page', 15);
            $perPage = min($perPage, 100); // Maximum 100 records per page
            $users = $query->paginate($perPage);

            // ✅ Full public URL provide karein for ALL users in index
            $users->getCollection()->transform(function ($user) {
                return $this->formatUserResponse($user);
            });

            return response()->json([
                'success' => true,
                'data' => $users->items(),
                'pagination' => [
                    'current_page' => $users->currentPage(),
                    'per_page' => $users->perPage(),
                    'total' => $users->total(),
                    'last_page' => $users->lastPage(),
                    'from' => $users->firstItem(),
                    'to' => $users->lastItem(),
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('Failed to fetch users: ' . $e->getMessage(), [
                'exception' => $e,
                'request_params' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch users',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Format user response with full profile picture URL
     */
    private function formatUserResponse($user)
    {
        if ($user->profile_picture_url) {
            // ✅ Full public URL provide karein
            $user->profile_picture_url = url('storage/' . $user->profile_picture_url);
        }

        // ✅ Ensure other relationships are properly loaded
        if (!$user->relationLoaded('role')) {
            $user->load('role');
        }
        if (!$user->relationLoaded('addresses')) {
            $user->load('addresses');
        }
        if (!$user->relationLoaded('userLanguages')) {
            $user->load('userLanguages');
        }

        return $user;
    }
    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $user = User::with([
                'role',
                'addresses',
                'userLanguages',
                'babysitterProfile',
                'parentProfile'
            ])->find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            // ✅ Full public URL provide karein
            if ($user->profile_picture_url) {
                $user->profile_picture_url = url('storage/' . $user->profile_picture_url);
            }

            return response()->json([
                'success' => true,
                'data' => $user
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch user',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get authenticated user profile
     */
    public function getProfile(Request $request): JsonResponse
    {
        try {
            $user = $request->user()->load([
                'role',
                'addresses',
                'userLanguages',
                'babysitterProfile',
                'parentProfile'
            ]);

            // ✅ Full public URL provide karein
            if ($user->profile_picture_url) {
                $user->profile_picture_url = url('storage/' . $user->profile_picture_url);
            }

            return response()->json([
                'success' => true,
                'data' => $user
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch user profile',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update authenticated user profile
     */
    public function updateProfile(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            $validator = Validator::make($request->all(), [
                'first_name' => 'sometimes|string|max:100',
                'last_name' => 'sometimes|string|max:100',
                'phone_number' => 'sometimes|string|max:20|unique:users,phone_number,' . $user->user_id . ',user_id',
                'date_of_birth' => 'sometimes|date|before:today',
                'gender' => 'sometimes|in:male,female,other,prefer_not_to_say',
                'bio' => 'nullable|string'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();
            $user->update($validated);

            // ✅ Full public URL provide karein
            if ($user->profile_picture_url) {
                $user->profile_picture_url = url('storage/' . $user->profile_picture_url);
            }

            return response()->json([
                'success' => true,
                'message' => 'Profile updated successfully',
                'data' => $user->fresh()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update profile',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update password for authenticated user
     */
    public function updateMyPassword(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            $validator = Validator::make($request->all(), [
                'current_password' => 'required|string',
                'new_password' => 'required|string|min:8|confirmed',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            if (!Hash::check($validated['current_password'], $user->password_hash)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Current password is incorrect'
                ], 422);
            }

            $user->update([
                'password_hash' => Hash::make($validated['new_password'])
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Password changed successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to change password',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update profile picture for authenticated user
     */
    public function updateMyProfilePicture(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            $validator = Validator::make($request->all(), [
                'profile_picture' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Delete old profile picture if exists
            if ($user->profile_picture_url) {
                $oldFilePath = public_path('storage/' . $user->profile_picture_url);
                if (file_exists($oldFilePath)) {
                    unlink($oldFilePath);
                }
            }

            // Store new profile picture
            $file = $request->file('profile_picture');
            $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();

            // ✅ Store in: public/storage/profile-pictures/
            $file->move(public_path('storage/profile-pictures'), $fileName);

            // ✅ Database mein relative path store karein
            $relativePath = 'profile-pictures/' . $fileName;
            $user->update(['profile_picture_url' => $relativePath]);

            // ✅ Fresh user data with full URL
            $user->refresh();

            return response()->json([
                'success' => true,
                'message' => 'Profile picture updated successfully',
                'data' => [
                    'profile_picture_url' => url('storage/' . $relativePath), // Full URL
                    'user' => $this->formatUserResponse($user)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update profile picture',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id): JsonResponse
    {
        try {
            $user = User::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            // ✅ STRICT PROTECTION: Prevent super admin (role_id = 1) deletion
            if ($user->role_id === 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Super admin account cannot be deleted. This is a protected system account.'
                ], 422);
            }

            // Prevent self-deletion
            if ($request->user()->user_id == $id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete your own account'
                ], 422);
            }

            $user->delete();

            return response()->json([
                'success' => true,
                'message' => 'User deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete user',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Deactivate user account
     */
    public function deactivateUser(Request $request, string $id): JsonResponse
    {
        try {
            $user = User::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            // ✅ STRICT PROTECTION: Prevent super admin (role_id = 1) deactivation
            if ($user->role_id === 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Super admin account cannot be deactivated. This is a protected system account.'
                ], 422);
            }

            // Prevent self-deactivation
            if ($request->user()->user_id == $id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot deactivate your own account'
                ], 422);
            }

            $user->deactivate();

            return response()->json([
                'success' => true,
                'message' => 'User deactivated successfully',
                'data' => $user->fresh()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to deactivate user',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }


    /**
     * Update user profile picture
     */
    public function updateProfilePicture(Request $request, string $id): JsonResponse
    {
        try {
            $user = User::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'profile_picture' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120' // 5MB max
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Delete old profile picture if exists
            if ($user->profile_picture_url && file_exists(public_path('storage/' . $user->profile_picture_url))) {
                unlink(public_path('storage/' . $user->profile_picture_url));
            }

            // Store new profile picture - PUBLIC/STORAGE mein save karein
            $file = $request->file('profile_picture');
            $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();

            // ✅ PUBLIC/STORAGE/PROFILE-PICTURES mein save karein
            $file->move(public_path('storage/profile-pictures'), $fileName);
            $profilePictureUrl = 'profile-pictures/' . $fileName;

            $user->update(['profile_picture_url' => $profilePictureUrl]);

            // ✅ Full public URL provide karein
            $user->profile_picture_url = url('storage/' . $profilePictureUrl);

            return response()->json([
                'success' => true,
                'message' => 'Profile picture updated successfully',
                'data' => [
                    'profile_picture_url' => $user->profile_picture_url,
                    'user' => $user->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update profile picture',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Delete profile picture for authenticated user
     */
    public function deleteMyProfilePicture(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            // Check if user has a profile picture
            if (!$user->profile_picture_url) {
                return response()->json([
                    'success' => false,
                    'message' => 'No profile picture found to delete'
                ], 404);
            }

            // Delete file from storage
            $filePath = public_path('storage/' . $user->profile_picture_url);

            if (file_exists($filePath)) {
                unlink($filePath);
            }

            // Update user record
            $user->update(['profile_picture_url' => null]);

            return response()->json([
                'success' => true,
                'message' => 'Profile picture deleted successfully',
                'data' => [
                    'user' => $user->fresh(['role', 'addresses', 'userLanguages'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete profile picture',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Delete profile picture for any user (admin functionality)
     */
    public function deleteProfilePicture(string $id): JsonResponse
    {
        try {
            $user = User::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            // Check if user has a profile picture
            if (!$user->profile_picture_url) {
                return response()->json([
                    'success' => false,
                    'message' => 'No profile picture found to delete'
                ], 404);
            }

            // Delete file from storage
            $filePath = public_path('storage/' . $user->profile_picture_url);

            if (file_exists($filePath)) {
                unlink($filePath);
            }

            // Update user record
            $user->update(['profile_picture_url' => null]);

            return response()->json([
                'success' => true,
                'message' => 'Profile picture deleted successfully',
                'data' => [
                    'user' => $user->fresh(['role', 'addresses', 'userLanguages'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete profile picture',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
