<?php

namespace App\Http\Controllers;

use App\Models\Report;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ReportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = Report::with(['generator']);

            // Filter by report type
            if ($request->has('type')) {
                $query->where('report_type', $request->get('type'));
            }

            // Filter by archived status
            if ($request->has('archived')) {
                $query->where('is_archived', $request->get('archived'));
            } else {
                $query->active(); // Default to non-archived reports
            }

            // Search by report name
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where('report_name', 'like', "%{$search}%");
            }

            // Filter by date range
            if ($request->has('date_range_start') && $request->has('date_range_end')) {
                $query->dateRange($request->get('date_range_start'), $request->get('date_range_end'));
            }

            // Order by
            $orderBy = $request->get('order_by', 'created_at');
            $orderDirection = $request->get('order_direction', 'desc');
            $query->orderBy($orderBy, $orderDirection);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $reports = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $reports->items(),
                'pagination' => [
                    'current_page' => $reports->currentPage(),
                    'per_page' => $reports->perPage(),
                    'total' => $reports->total(),
                    'last_page' => $reports->lastPage(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch reports',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $user = $request->user();
            $validated = $request->validate([
                'report_name' => 'required|string|max:255',
                'report_type' => 'required|in:financial,user,booking,commission,performance,system',
                'report_data' => 'required|array',
                'date_range_start' => 'nullable|date',
                'date_range_end' => 'nullable|date|after_or_equal:date_range_start'
            ]);

            $validated['generated_by'] = $user->user_id;
            $report = Report::create($validated);

            return response()->json([
                'success' => true,
                'message' => 'Report created successfully',
                'data' => $report->load('generator')
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $report = Report::with(['generator'])->find($id);

            if (!$report) {
                return response()->json([
                    'success' => false,
                    'message' => 'Report not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $report
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $report = Report::find($id);

            if (!$report) {
                return response()->json([
                    'success' => false,
                    'message' => 'Report not found'
                ], 404);
            }

            $validated = $request->validate([
                'report_name' => 'sometimes|string|max:255',
                'report_type' => 'sometimes|in:financial,user,booking,commission,performance,system',
                'report_data' => 'sometimes|array',
                'date_range_start' => 'nullable|date',
                'date_range_end' => 'nullable|date|after_or_equal:date_range_start'
            ]);

            $report->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Report updated successfully',
                'data' => $report->load('generator')
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $report = Report::find($id);

            if (!$report) {
                return response()->json([
                    'success' => false,
                    'message' => 'Report not found'
                ], 404);
            }

            $report->delete();

            return response()->json([
                'success' => true,
                'message' => 'Report deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Archive report
     */
    public function archive(string $id): JsonResponse
    {
        try {
            $report = Report::find($id);

            if (!$report) {
                return response()->json([
                    'success' => false,
                    'message' => 'Report not found'
                ], 404);
            }

            $report->archive();

            return response()->json([
                'success' => true,
                'message' => 'Report archived successfully',
                'data' => $report
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to archive report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Unarchive report
     */
    public function unarchive(string $id): JsonResponse
    {
        try {
            $report = Report::find($id);

            if (!$report) {
                return response()->json([
                    'success' => false,
                    'message' => 'Report not found'
                ], 404);
            }

            $report->unarchive();

            return response()->json([
                'success' => true,
                'message' => 'Report unarchived successfully',
                'data' => $report
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to unarchive report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Generate financial report
     */
    public function generateFinancialReport(Request $request): JsonResponse
    {
        try {
            $user = $request->user();
            $validated = $request->validate([
                'date_range_start' => 'required|date',
                'date_range_end' => 'required|date|after_or_equal:date_range_start',
                'include_monthly_breakdown' => 'boolean|default:true',
                'include_category_breakdown' => 'boolean|default:true'
            ]);

            // Here you would call your stored procedure or generate the report data
            $reportData = $this->generateFinancialData($validated);

            $report = Report::create([
                'report_name' => 'Financial Report - ' . $validated['date_range_start'] . ' to ' . $validated['date_range_end'],
                'report_type' => 'financial',
                'report_data' => $reportData,
                'date_range_start' => $validated['date_range_start'],
                'date_range_end' => $validated['date_range_end'],
                'generated_by' => $user->user_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Financial report generated successfully',
                'data' => $report->load('generator')
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate financial report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Generate user report
     */
    public function generateUserReport(Request $request): JsonResponse
    {
        try {
            $user = $request->user();
            $validated = $request->validate([
                'date_range_start' => 'required|date',
                'date_range_end' => 'required|date|after_or_equal:date_range_start',
                'include_registration_trends' => 'boolean|default:true',
                'include_user_metrics' => 'boolean|default:true'
            ]);

            // Generate user report data
            $reportData = $this->generateUserData($validated);

            $report = Report::create([
                'report_name' => 'User Registration Report - ' . $validated['date_range_start'] . ' to ' . $validated['date_range_end'],
                'report_type' => 'user',
                'report_data' => $reportData,
                'date_range_start' => $validated['date_range_start'],
                'date_range_end' => $validated['date_range_end'],
                'generated_by' => $user->user_id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'User report generated successfully',
                'data' => $report->load('generator')
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate user report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Generate financial data (example implementation)
     */
    private function generateFinancialData(array $params): array
    {
        // This would typically call your stored procedure or generate actual data
        // For now, returning sample data
        return [
            'summary' => [
                'total_revenue' => 150000.00,
                'total_commissions' => 22500.00,
                'total_bookings' => 450,
                'avg_booking_value' => 333.33
            ],
            'monthly_breakdown' => $params['include_monthly_breakdown'] ? [
                ['month' => '2024-01', 'revenue' => 50000.00, 'commissions' => 7500.00, 'bookings' => 150],
                ['month' => '2024-02', 'revenue' => 55000.00, 'commissions' => 8250.00, 'bookings' => 165],
                ['month' => '2024-03', 'revenue' => 45000.00, 'commissions' => 6750.00, 'bookings' => 135]
            ] : null,
            'category_breakdown' => $params['include_category_breakdown'] ? [
                ['category' => 'Full-time Nanny', 'revenue' => 75000.00, 'commissions' => 11250.00, 'bookings' => 225],
                ['category' => 'Part-time Nanny', 'revenue' => 45000.00, 'commissions' => 6750.00, 'bookings' => 135],
                ['category' => 'Elderly Care', 'revenue' => 30000.00, 'commissions' => 4500.00, 'bookings' => 90]
            ] : null,
            'generated_at' => now()->toISOString(),
            'parameters_used' => $params
        ];
    }

    /**
     * Generate user data (example implementation)
     */
    private function generateUserData(array $params): array
    {
        return [
            'summary' => [
                'total_users' => 1500,
                'total_parents' => 900,
                'total_babysitters' => 600,
                'verified_users' => 1200,
                'active_users' => 1400
            ],
            'registration_trends' => $params['include_registration_trends'] ? [
                ['date' => '2024-01-01', 'parents' => 30, 'babysitters' => 20, 'total' => 50],
                ['date' => '2024-01-02', 'parents' => 25, 'babysitters' => 15, 'total' => 40],
                ['date' => '2024-01-03', 'parents' => 35, 'babysitters' => 25, 'total' => 60]
            ] : null,
            'user_metrics' => $params['include_user_metrics'] ? [
                'completion_rate' => 85.5,
                'verification_rate' => 80.0,
                'active_rate' => 93.3
            ] : null,
            'generated_at' => now()->toISOString(),
            'parameters_used' => $params
        ];
    }
}
