<?php

namespace App\Http\Controllers;

use App\Models\Payout;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class PayoutController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = Payout::with(['babysitter', 'payment']);

            // Filter by payout status
            if ($request->has('status')) {
                $query->where('payout_status', $request->get('status'));
            }

            // Filter by babysitter
            if ($request->has('babysitter_id')) {
                $query->where('babysitter_id', $request->get('babysitter_id'));
            }

            // Date range filter
            if ($request->has('start_date') && $request->has('end_date')) {
                $query->whereBetween('created_at', [
                    $request->get('start_date'),
                    $request->get('end_date')
                ]);
            }

            // Order by
            $orderBy = $request->get('order_by', 'created_at');
            $orderDirection = $request->get('order_direction', 'desc');
            $query->orderBy($orderBy, $orderDirection);

            // Pagination
            $perPage = $request->get('per_page', 10);
            $payouts = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $payouts->items(),
                'pagination' => [
                    'current_page' => $payouts->currentPage(),
                    'per_page' => $payouts->perPage(),
                    'total' => $payouts->total(),
                    'last_page' => $payouts->lastPage(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payouts',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'babysitter_id' => 'required|exists:users,user_id',
                'payment_id' => 'required|exists:payments,payment_id',
                'amount' => 'required|numeric|min:0',
                'payout_method' => 'required|in:bank_transfer,paypal,interac,check,cash',
                'payout_reference' => 'nullable|string|max:255',
                'payout_notes' => 'nullable|string',
                'scheduled_payout_date' => 'required|date'
            ]);

            $payout = Payout::create($validated);

            return response()->json([
                'success' => true,
                'message' => 'Payout created successfully',
                'data' => $payout
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create payout',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $payout = Payout::with(['babysitter', 'payment'])->find($id);

            if (!$payout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payout not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $payout
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payout',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $payout = Payout::find($id);

            if (!$payout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payout not found'
                ], 404);
            }

            $validated = $request->validate([
                'payout_status' => 'sometimes|in:pending,processing,completed,failed',
                'payout_reference' => 'nullable|string|max:255',
                'payout_notes' => 'nullable|string',
                'scheduled_payout_date' => 'sometimes|date'
            ]);

            $payout->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Payout updated successfully',
                'data' => $payout
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payout',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $payout = Payout::find($id);

            if (!$payout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payout not found'
                ], 404);
            }

            // Only allow deletion of pending payouts
            if (!$payout->isPending()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete processing or completed payouts'
                ], 422);
            }

            $payout->delete();

            return response()->json([
                'success' => true,
                'message' => 'Payout deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete payout',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get my payouts (for authenticated babysitter)
     */
    public function getMyPayouts(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            if ($user->user_type !== 'babysitter') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only babysitters can access payouts'
                ], 403);
            }

            $query = Payout::with(['payment'])->where('babysitter_id', $user->user_id);

            // Filter by status
            if ($request->has('status')) {
                $query->where('payout_status', $request->get('status'));
            }

            // Order by
            $orderBy = $request->get('order_by', 'created_at');
            $orderDirection = $request->get('order_direction', 'desc');
            $query->orderBy($orderBy, $orderDirection);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $payouts = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $payouts->items(),
                'pagination' => [
                    'current_page' => $payouts->currentPage(),
                    'per_page' => $payouts->perPage(),
                    'total' => $payouts->total(),
                    'last_page' => $payouts->lastPage(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payouts',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Mark payout as processing
     */
    public function markAsProcessing(string $id): JsonResponse
    {
        try {
            $payout = Payout::find($id);

            if (!$payout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payout not found'
                ], 404);
            }

            $payout->markAsProcessing();

            return response()->json([
                'success' => true,
                'message' => 'Payout marked as processing',
                'data' => $payout
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payout status',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Mark payout as completed
     */
    public function markAsCompleted(Request $request, string $id): JsonResponse
    {
        try {
            $payout = Payout::find($id);

            if (!$payout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payout not found'
                ], 404);
            }

            $payout->markAsCompleted($request->get('payout_reference'));

            return response()->json([
                'success' => true,
                'message' => 'Payout marked as completed',
                'data' => $payout
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to complete payout',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Mark payout as failed
     */
    public function markAsFailed(Request $request, string $id): JsonResponse
    {
        try {
            $payout = Payout::find($id);

            if (!$payout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payout not found'
                ], 404);
            }

            $payout->markAsFailed($request->get('notes'));

            return response()->json([
                'success' => true,
                'message' => 'Payout marked as failed',
                'data' => $payout
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark payout as failed',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
