<?php

namespace App\Http\Controllers;

use App\Models\OccupationCategory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class OccupationCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */

    /**
 * Get all occupation categories without pagination
 */
public function getAll(): JsonResponse
{
    try {
        $categories = OccupationCategory::orderBy('category_name', 'asc')->get();

        return response()->json([
            'success' => true,
            'data' => $categories
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch all occupation categories',
            'error' => config('app.debug') ? $e->getMessage() : null
        ], 500);
    }
}

    public function index(Request $request): JsonResponse
    {
        try {
            $query = OccupationCategory::query();

            // Filter by active status
            if ($request->has('active_only') && $request->boolean('active_only')) {
                $query->active();
            }

            // Filter by category type
            if ($request->has('category_type')) {
                $query->where('category_type', $request->get('category_type'));
            }

            // Search by category name
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where('category_name', 'like', "%{$search}%")
                      ->orWhere('category_description', 'like', "%{$search}%");
            }

            // Order by
            $orderBy = $request->get('order_by', 'category_name');
            $orderDirection = $request->get('order_direction', 'asc');
            $query->orderBy($orderBy, $orderDirection);

            // Get results (with or without pagination)
            if ($request->has('paginate') && $request->boolean('paginate')) {
                $perPage = $request->get('per_page', 15);
                $categories = $query->paginate($perPage);

                return response()->json([
                    'success' => true,
                    'data' => $categories->items(),
                    'pagination' => [
                        'current_page' => $categories->currentPage(),
                        'per_page' => $categories->perPage(),
                        'total' => $categories->total(),
                        'last_page' => $categories->lastPage(),
                    ]
                ]);
            } else {
                $categories = $query->get();

                return response()->json([
                    'success' => true,
                    'data' => $categories
                ]);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch occupation categories',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'category_name' => 'required|string|max:255|unique:occupation_categories,category_name',
                'category_description' => 'nullable|string',
                'category_type' => 'required|in:nanny,home_worker,elderly_care,pet_care,house_helper,special_needs_care',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            $category = OccupationCategory::create($validated);

            return response()->json([
                'success' => true,
                'message' => 'Occupation category created successfully',
                'data' => $category
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create occupation category',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $category = OccupationCategory::with('babysitterProfiles')->find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Occupation category not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $category
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch occupation category',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $category = OccupationCategory::find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Occupation category not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'category_name' => 'sometimes|string|max:255|unique:occupation_categories,category_name,' . $id . ',category_id',
                'category_description' => 'nullable|string',
                'category_type' => 'sometimes|in:nanny,home_worker,elderly_care,pet_care,house_helper,special_needs_care',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            $category->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Occupation category updated successfully',
                'data' => $category
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update occupation category',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $category = OccupationCategory::find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Occupation category not found'
                ], 404);
            }

            // Check if category has associated babysitter profiles
            if ($category->babysitterProfiles()->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete category with associated babysitter profiles'
                ], 422);
            }

            $category->delete();

            return response()->json([
                'success' => true,
                'message' => 'Occupation category deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete occupation category',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get categories by type
     */
    public function getByType(string $type): JsonResponse
    {
        try {
            $categories = OccupationCategory::where('category_type', $type)
                ->active()
                ->orderBy('category_name')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $categories
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch categories by type',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get active categories for dropdown
     */
    public function getActiveCategories(): JsonResponse
    {
        try {
            $categories = OccupationCategory::active()
                ->orderBy('category_name')
                ->select('category_id', 'category_name', 'category_type')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $categories
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch active categories',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
