<?php

namespace App\Http\Controllers;

use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class LanguageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = Language::query();

            // Filter by active status
            if ($request->has('active_only') && $request->boolean('active_only')) {
                $query->where('is_active', true);
            }

            // Search by name or code
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where(function($q) use ($search) {
                    $q->where('language_name', 'like', "%{$search}%")
                      ->orWhere('language_code', 'like', "%{$search}%");
                });
            }

            // Order by
            $orderBy = $request->get('order_by', 'language_name');
            $orderDirection = $request->get('order_direction', 'asc');
            $query->orderBy($orderBy, $orderDirection);

            // Get all or paginate
            if ($request->has('paginate') && !$request->boolean('paginate')) {
                $languages = $query->get();
            } else {
                $perPage = $request->get('per_page', 10);
                $languages = $query->paginate($perPage);
            }

            return response()->json([
                'success' => true,
                'data' => $languages
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch languages',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
 * Get all languages without pagination
 */
public function getAllLanguages(Request $request): JsonResponse
{
    try {
        $query = Language::query();

        // Filter by active status
        if ($request->has('active_only') && $request->boolean('active_only')) {
            $query->where('is_active', true);
        }

        // Search by name or code
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function($q) use ($search) {
                $q->where('language_name', 'like', "%{$search}%")
                  ->orWhere('language_code', 'like', "%{$search}%");
            });
        }

        // Order by
        $orderBy = $request->get('order_by', 'language_name');
        $orderDirection = $request->get('order_direction', 'asc');
        $query->orderBy($orderBy, $orderDirection);

        // Get all data without pagination
        $languages = $query->get();

        return response()->json([
            'success' => true,
            'data' => $languages,
            'count' => $languages->count()
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch all languages',
            'error' => config('app.debug') ? $e->getMessage() : null
        ], 500);
    }
}

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            // Simple authentication check only
            if (!$request->user()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthenticated'
                ], 401);
            }

            $validator = Validator::make($request->all(), [
                'language_code' => 'required|string|max:10|unique:languages,language_code',
                'language_name' => 'required|string|max:100',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $language = Language::create($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Language created successfully',
                'data' => $language
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create language',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, string $id): JsonResponse
    {
        try {
            $language = Language::find($id);

            if (!$language) {
                return response()->json([
                    'success' => false,
                    'message' => 'Language not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $language
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch language',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            // Simple authentication check only
            if (!$request->user()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthenticated'
                ], 401);
            }

            $language = Language::find($id);

            if (!$language) {
                return response()->json([
                    'success' => false,
                    'message' => 'Language not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'language_code' => 'sometimes|string|max:10|unique:languages,language_code,' . $id . ',language_id',
                'language_name' => 'sometimes|string|max:100',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $language->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Language updated successfully',
                'data' => $language
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update language',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id): JsonResponse
    {
        try {
            // Simple authentication check only
            if (!$request->user()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthenticated'
                ], 401);
            }

            $language = Language::find($id);

            if (!$language) {
                return response()->json([
                    'success' => false,
                    'message' => 'Language not found'
                ], 404);
            }

            // Check if language is being used
            if ($language->userLanguages()->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete language that is in use by users'
                ], 422);
            }

            $language->delete();

            return response()->json([
                'success' => true,
                'message' => 'Language deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete language',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get active languages list
     */
    public function getActiveLanguages(): JsonResponse
    {
        try {
            $languages = Language::where('is_active', true)
                ->orderBy('language_name')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $languages
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch active languages',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get language by code
     */
    public function getByCode(string $code): JsonResponse
    {
        try {
            $language = Language::where('language_code', $code)
                ->where('is_active', true)
                ->first();

            if (!$language) {
                return response()->json([
                    'success' => false,
                    'message' => 'Language not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $language
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch language',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
