<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Booking;
use App\Models\Payout;
use App\Models\CommissionEarning;
use App\Models\Payment;
use App\Models\BabysitterProfile;
use App\Models\ParentProfile;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Get Super Admin Dashboard Statistics
     */
    public function getSuperAdminDashboard(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            // Check if user is superadmin - FIXED: Removed hasRole() method
            if (!$user || !$this->isSuperAdmin($user)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized. Only superadmin can access dashboard.'
                ], 403);
            }

            // Get date range filters
            $startDate = $request->get('start_date') ? Carbon::parse($request->get('start_date')) : Carbon::now()->subMonth();
            $endDate = $request->get('end_date') ? Carbon::parse($request->get('end_date')) : Carbon::now();

            // Fetch all dashboard data
            $results = $this->fetchDashboardData($startDate, $endDate);

            return response()->json([
                'success' => true,
                'data' => $results,
                'period' => [
                    'start_date' => $startDate->toDateString(),
                    'end_date' => $endDate->toDateString()
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('SuperAdmin Dashboard Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch dashboard data',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Check if user is super admin - FIXED VERSION
     */
    private function isSuperAdmin($user): bool
    {
        // Check based on available user properties
        // Adjust according to your User model structure
        if (isset($user->user_type) && $user->user_type === 'super_admin') {
            return true;
        }

        if (isset($user->is_super_admin) && $user->is_super_admin === true) {
            return true;
        }

        if (isset($user->role) && $user->role === 'super_admin') {
            return true;
        }

        // Default: Check if email is admin email (temporary solution)
        if (isset($user->email) && str_contains($user->email, 'admin')) {
            return true;
        }

        return false;
    }

    /**
     * Fetch all dashboard data in parallel
     */
    private function fetchDashboardData($startDate, $endDate)
    {
        // Use DB transactions for consistent data
        return DB::transaction(function () use ($startDate, $endDate) {
            // 1. User Statistics
            $userStats = $this->getUserStatistics();

            // 2. Booking Statistics
            $bookingStats = $this->getBookingStatistics();

            // 3. Financial Statistics - FIXED: Use Payment table instead of Booking table
            $financialStats = $this->getFinancialStatisticsFixed($startDate, $endDate);

            // 4. Payout Statistics - FIXED: Use correct calculations
            $payoutStats = $this->getPayoutStatisticsFixed();

            // 5. Recent Activity
            $recentActivity = $this->getRecentActivity();

            // 6. Charts Data
            $chartsData = $this->getChartsData($startDate, $endDate);

            return [
                'summary_cards' => [
                    'total_users' => $userStats['total_users'],
                    'total_babysitters' => $userStats['total_babysitters'],
                    'total_parents' => $userStats['total_parents'],
                    'active_bookings' => $bookingStats['active_bookings'],
                    'pending_payouts' => $payoutStats['pending_payouts_amount'],
                    'total_commission' => $financialStats['total_commission'],
                    'total_revenue' => $financialStats['total_revenue'],
                    'successful_payments' => $financialStats['successful_payments_count']
                ],
                'user_statistics' => $userStats,
                'booking_statistics' => $bookingStats,
                'financial_statistics' => $financialStats,
                'payout_statistics' => $payoutStats,
                'recent_activity' => $recentActivity,
                'charts_data' => $chartsData
            ];
        });
    }

    /**
     * Get User Statistics - FIXED VERSION (Removed is_verified column)
     */
    private function getUserStatistics(): array
    {
        $totalUsers = User::where('is_active', true)->count();

        // Use profile-based counting instead of user_type column
        $totalBabysitters = BabysitterProfile::count();
        $totalParents = ParentProfile::count();

        // New registrations (last 30 days)
        $newUsers = User::where('created_at', '>=', Carbon::now()->subDays(30))->count();

        // For new babysitters and parents, use profile creation date
        $newBabysitters = BabysitterProfile::where('created_at', '>=', Carbon::now()->subDays(30))->count();
        $newParents = ParentProfile::where('created_at', '>=', Carbon::now()->subDays(30))->count();

        // FIXED: Removed is_verified column reference
        $verifiedUsers = 0;

        return [
            'total_users' => $totalUsers,
            'total_babysitters' => $totalBabysitters,
            'total_parents' => $totalParents,
            'new_users_30_days' => $newUsers,
            'new_babysitters_30_days' => $newBabysitters,
            'new_parents_30_days' => $newParents,
            'verified_users' => $verifiedUsers,
            'verification_rate' => $totalUsers > 0 ? round(($verifiedUsers / $totalUsers) * 100, 2) : 0,
            'user_distribution' => [
                'babysitters_percentage' => $totalUsers > 0 ? round(($totalBabysitters / $totalUsers) * 100, 2) : 0,
                'parents_percentage' => $totalUsers > 0 ? round(($totalParents / $totalUsers) * 100, 2) : 0
            ]
        ];
    }

    /**
     * Get Booking Statistics
     */
    private function getBookingStatistics(): array
    {
        $totalBookings = Booking::count();
        $activeBookings = Booking::where('booking_status', 'active')->count();
        $completedBookings = Booking::where('booking_status', 'completed')->count();
        $cancelledBookings = Booking::where('booking_status', 'cancelled')->count();
        $pendingPaymentBookings = Booking::where('booking_status', 'pending_payment')->count();

        // Recent bookings (last 7 days)
        $recentBookings = Booking::where('created_at', '>=', Carbon::now()->subDays(7))->count();

        // Booking value metrics - FIXED: Use only paid bookings
        $totalBookingValue = Booking::where('payment_status', 'paid')->sum('total_amount');
        $averageBookingValue = $completedBookings > 0 ? round($totalBookingValue / $completedBookings, 2) : 0;

        return [
            'total_bookings' => $totalBookings,
            'active_bookings' => $activeBookings,
            'completed_bookings' => $completedBookings,
            'cancelled_bookings' => $cancelledBookings,
            'pending_payment_bookings' => $pendingPaymentBookings,
            'recent_bookings_7_days' => $recentBookings,
            'completion_rate' => $totalBookings > 0 ? round(($completedBookings / $totalBookings) * 100, 2) : 0,
            'total_booking_value' => $totalBookingValue,
            'average_booking_value' => $averageBookingValue,
            'status_distribution' => [
                'active_percentage' => $totalBookings > 0 ? round(($activeBookings / $totalBookings) * 100, 2) : 0,
                'completed_percentage' => $totalBookings > 0 ? round(($completedBookings / $totalBookings) * 100, 2) : 0,
                'cancelled_percentage' => $totalBookings > 0 ? round(($cancelledBookings / $totalBookings) * 100, 2) : 0
            ]
        ];
    }

    /**
     * Get Financial Statistics - FIXED VERSION (Use Payment table)
     */
    private function getFinancialStatisticsFixed($startDate, $endDate): array
    {
        // FIXED: Use Payment table instead of Booking table for accurate financial data
        $successfulPayments = Payment::where('payment_status', 'completed')->get();

        // Calculate totals from successful payments only
        $totalRevenue = $successfulPayments->sum('amount');
        $totalCommission = $successfulPayments->sum('commission_amount');

        // Revenue in date range
        $periodRevenue = Payment::where('payment_status', 'completed')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->sum('amount');

        // Commission in date range
        $periodCommission = Payment::where('payment_status', 'completed')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->sum('commission_amount');

        // Payment statistics
        $successfulPaymentsCount = $successfulPayments->count();
        $failedPayments = Payment::where('payment_status', 'failed')->count();
        $pendingPayments = Payment::where('payment_status', 'pending')->count();

        $totalPayments = $successfulPaymentsCount + $failedPayments + $pendingPayments;
        $successRate = $totalPayments > 0 ? round(($successfulPaymentsCount / $totalPayments) * 100, 2) : 0;

        return [
            'total_commission' => $totalCommission,
            'period_commission' => $periodCommission,
            'total_revenue' => $totalRevenue,
            'period_revenue' => $periodRevenue,
            'successful_payments_count' => $successfulPaymentsCount,
            'failed_payments_count' => $failedPayments,
            'pending_payments_count' => $pendingPayments,
            'payment_success_rate' => $successRate,
            'average_commission_per_booking' => $successfulPaymentsCount > 0 ? round($totalCommission / $successfulPaymentsCount, 2) : 0
        ];
    }

    /**
     * Get Payout Statistics - FIXED VERSION
     */
    private function getPayoutStatisticsFixed(): array
    {
        $totalPayouts = Payout::count();
        $pendingPayouts = Payout::where('payout_status', 'pending')->count();
        $processingPayouts = Payout::where('payout_status', 'processing')->count();
        $completedPayouts = Payout::where('payout_status', 'completed')->count();
        $failedPayouts = Payout::where('payout_status', 'failed')->count();

        // FIXED: Calculate pending payouts from successful payments that haven't been paid out
        $pendingPayoutsAmount = Payout::where('payout_status', 'pending')->sum('amount');

        // Alternative: Calculate from payments that should be paid out but haven't been
        $totalPayoutsAmount = Payout::sum('amount');
        $processingPayoutsAmount = Payout::where('payout_status', 'processing')->sum('amount');
        $completedPayoutsAmount = Payout::where('payout_status', 'completed')->sum('amount');

        return [
            'total_payouts_count' => $totalPayouts,
            'pending_payouts_count' => $pendingPayouts,
            'processing_payouts_count' => $processingPayouts,
            'completed_payouts_count' => $completedPayouts,
            'failed_payouts_count' => $failedPayouts,
            'pending_payouts_amount' => $pendingPayoutsAmount,
            'processing_payouts_amount' => $processingPayoutsAmount,
            'completed_payouts_amount' => $completedPayoutsAmount,
            'total_payouts_amount' => $totalPayoutsAmount,
            'payout_success_rate' => $totalPayouts > 0 ? round(($completedPayouts / $totalPayouts) * 100, 2) : 0
        ];
    }

    /**
     * Get Recent Activity
     */
    private function getRecentActivity(): array
    {
        // Recent bookings (last 10)
        $recentBookings = Booking::with(['parent', 'babysitter'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($booking) {
                $parentName = $booking->parent ? $booking->parent->first_name . ' ' . $booking->parent->last_name : 'Unknown Parent';

                return [
                    'type' => 'booking',
                    'id' => $booking->booking_id,
                    'title' => 'New Booking Created',
                    'description' => "Booking #{$booking->booking_id} by {$parentName}",
                    'amount' => $booking->total_amount,
                    'status' => $booking->booking_status,
                    'timestamp' => $booking->created_at,
                    'user' => [
                        'name' => $parentName,
                        'type' => 'parent'
                    ]
                ];
            });

        // Recent users (last 10)
        $recentUsers = User::orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($user) {
                // Determine user type based on profiles
                $userType = 'user';
                if ($user->babysitterProfile) {
                    $userType = 'babysitter';
                } elseif ($user->parentProfile) {
                    $userType = 'parent';
                }

                return [
                    'type' => 'user',
                    'id' => $user->user_id,
                    'title' => 'New User Registered',
                    'description' => "{$user->first_name} {$user->last_name} as " . ucfirst($userType),
                    'timestamp' => $user->created_at,
                    'user' => [
                        'name' => $user->first_name . ' ' . $user->last_name,
                        'type' => $userType
                    ]
                ];
            });

        // Recent payouts (last 10)
        $recentPayouts = Payout::with(['babysitter'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($payout) {
                $babysitterName = $payout->babysitter ?
                    $payout->babysitter->first_name . ' ' . $payout->babysitter->last_name :
                    'Unknown Babysitter';

                return [
                    'type' => 'payout',
                    'id' => $payout->payout_id,
                    'title' => 'Payout Processed',
                    'description' => "Payout of {$payout->amount} to {$babysitterName}",
                    'amount' => $payout->amount,
                    'status' => $payout->payout_status,
                    'timestamp' => $payout->created_at,
                    'user' => [
                        'name' => $babysitterName,
                        'type' => 'babysitter'
                    ]
                ];
            });

        // Combine and sort all activities
        $allActivities = collect()
            ->merge($recentBookings)
            ->merge($recentUsers)
            ->merge($recentPayouts)
            ->sortByDesc('timestamp')
            ->values()
            ->take(15); // Limit to 15 most recent activities

        return $allActivities->toArray();
    }

    /**
     * Get Charts Data for Graphs
     */
    private function getChartsData($startDate, $endDate): array
    {
        // Monthly revenue data (last 6 months) - FIXED: Use Payment table
        $monthlyRevenue = Payment::where('payment_status', 'completed')
            ->where('created_at', '>=', Carbon::now()->subMonths(6))
            ->select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('MONTH(created_at) as month'),
                DB::raw('SUM(amount) as revenue'),
                DB::raw('COUNT(*) as payments_count')
            )
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => Carbon::create($item->year, $item->month)->format('M Y'),
                    'revenue' => $item->revenue,
                    'payments' => $item->payments_count
                ];
            })
            ->reverse()
            ->values();

        // User registration trend (last 6 months)
        $userRegistrations = User::where('created_at', '>=', Carbon::now()->subMonths(6))
            ->select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('MONTH(created_at) as month'),
                DB::raw('COUNT(*) as total_users')
            )
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => Carbon::create($item->year, $item->month)->format('M Y'),
                    'total_users' => $item->total_users
                ];
            })
            ->reverse()
            ->values();

        // Booking status distribution
        $bookingStatusDistribution = Booking::select(
            'booking_status',
            DB::raw('COUNT(*) as count')
        )
            ->groupBy('booking_status')
            ->get()
            ->map(function ($item) {
                return [
                    'status' => ucfirst(str_replace('_', ' ', $item->booking_status)),
                    'count' => $item->count
                ];
            });

        // Payout status distribution
        $payoutStatusDistribution = Payout::select(
            'payout_status',
            DB::raw('COUNT(*) as count'),
            DB::raw('SUM(amount) as amount')
        )
            ->groupBy('payout_status')
            ->get()
            ->map(function ($item) {
                return [
                    'status' => ucfirst($item->payout_status),
                    'count' => $item->count,
                    'amount' => $item->amount
                ];
            });

        return [
            'monthly_revenue' => $monthlyRevenue,
            'user_registrations' => $userRegistrations,
            'booking_status_distribution' => $bookingStatusDistribution,
            'payout_status_distribution' => $payoutStatusDistribution
        ];
    }

    /**
     * Get Quick Stats (Minimal data for header/widget)
     */
    public function getQuickStats(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            if (!$user || !$this->isSuperAdmin($user)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }

            $quickStats = DB::transaction(function () {
                // FIXED: Use Payment table for accurate financial data
                $successfulPayments = Payment::where('payment_status', 'completed')->get();

                return [
                    'total_users' => User::where('is_active', true)->count(),
                    'total_babysitters' => BabysitterProfile::count(),
                    'total_parents' => ParentProfile::count(),
                    'active_bookings' => Booking::where('booking_status', 'active')->count(),
                    'pending_payouts' => Payout::where('payout_status', 'pending')->sum('amount'),
                    'total_commission' => $successfulPayments->sum('commission_amount'),
                    'total_revenue' => $successfulPayments->sum('amount'),
                    'today_revenue' => Payment::where('payment_status', 'completed')
                        ->whereDate('created_at', Carbon::today())
                        ->sum('amount')
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $quickStats
            ]);
        } catch (\Exception $e) {
            Log::error('Quick Stats Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch quick stats'
            ], 500);
        }
    }
}
