<?php
// app/Http/Controllers/CommissionRateController.php

namespace App\Http\Controllers;

use App\Models\CommissionRate;
use App\Models\OccupationCategory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class CommissionRateController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            // ✅ UPDATED: Include occupation category relationship
            $query = CommissionRate::with(['createdBy', 'occupationCategory']);

            // Filter by active status
            if ($request->has('active_only') && $request->boolean('active_only')) {
                $query->active();
            }

            // Filter by currently effective
            if ($request->has('currently_effective') && $request->boolean('currently_effective')) {
                $query->currentlyEffective();
            }

            // Filter by applicable_to
            if ($request->has('applicable_to')) {
                $query->where('applicable_to', $request->get('applicable_to'));
            }

            // ✅ NEW: Filter by occupation category
            if ($request->has('category_id')) {
                $categoryId = $request->get('category_id');
                if ($categoryId === 'null') {
                    $query->whereNull('category_id');
                } else {
                    $query->where('category_id', $categoryId);
                }
            }

            // Filter by commission type
            if ($request->has('commission_type')) {
                $query->where('commission_type', $request->get('commission_type'));
            }

            // Search by name
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where('commission_name', 'like', "%{$search}%");
            }

            // Order by
            $orderBy = $request->get('order_by', 'created_at');
            $orderDirection = $request->get('order_direction', 'desc');
            $query->orderBy($orderBy, $orderDirection);

            // Pagination
            $perPage = $request->get('per_page', 10);
            $commissionRates = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $commissionRates->items(),
                'pagination' => [
                    'current_page' => $commissionRates->currentPage(),
                    'per_page' => $commissionRates->perPage(),
                    'total' => $commissionRates->total(),
                    'last_page' => $commissionRates->lastPage(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch commission rates',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'commission_name' => 'required|string|max:255',
                'commission_type' => 'required|in:percentage,fixed',
                'commission_value' => 'required|numeric|min:0',
                'min_amount' => 'nullable|numeric|min:0',
                'max_amount' => 'nullable|numeric|min:0',
                'applicable_to' => 'nullable|in:all,nanny,home_worker,elderly_care,pet_care,house_helper,special_needs_care',
                'category_id' => 'nullable|exists:occupation_categories,category_id', // ✅ NEW VALIDATION
                'is_active' => 'boolean',
                'effective_from' => 'required|date',
                'effective_to' => 'nullable|date|after:effective_from'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();
            $validated['created_by'] = $request->user()->user_id;

            $commissionRate = CommissionRate::create($validated);
            // ✅ UPDATED: Load both relationships
            $commissionRate->load(['createdBy', 'occupationCategory']);

            return response()->json([
                'success' => true,
                'message' => 'Commission rate created successfully',
                'data' => $commissionRate
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create commission rate',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, string $id): JsonResponse
    {
        try {
            // ✅ UPDATED: Include occupation category
            $commissionRate = CommissionRate::with(['createdBy', 'occupationCategory'])->find($id);

            if (!$commissionRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission rate not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $commissionRate
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch commission rate',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $commissionRate = CommissionRate::find($id);

            if (!$commissionRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission rate not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'commission_name' => 'sometimes|string|max:255',
                'commission_type' => 'sometimes|in:percentage,fixed',
                'commission_value' => 'sometimes|numeric|min:0',
                'min_amount' => 'nullable|numeric|min:0',
                'max_amount' => 'nullable|numeric|min:0',
                'applicable_to' => 'sometimes|in:all,nanny,home_worker,elderly_care,pet_care,house_helper,special_needs_care',
                'category_id' => 'nullable|exists:occupation_categories,category_id', // ✅ NEW VALIDATION
                'is_active' => 'boolean',
                'effective_from' => 'sometimes|date',
                'effective_to' => 'nullable|date|after:effective_from'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $commissionRate->update($validator->validated());
            // ✅ UPDATED: Load both relationships
            $commissionRate->load(['createdBy', 'occupationCategory']);

            return response()->json([
                'success' => true,
                'message' => 'Commission rate updated successfully',
                'data' => $commissionRate
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update commission rate',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id): JsonResponse
    {
        try {
            $commissionRate = CommissionRate::find($id);

            if (!$commissionRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission rate not found'
                ], 404);
            }

            $commissionRate->delete();

            return response()->json([
                'success' => true,
                'message' => 'Commission rate deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete commission rate',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * ✅ UPDATED: Get commission rate for specific category
     */
    public function getRateFor(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'category_type' => 'required|string',
                'category_id' => 'nullable|exists:occupation_categories,category_id', // ✅ NEW
                'total_amount' => 'nullable|numeric|min:0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            // ✅ UPDATED: Use new method with category_id
            $commissionRate = CommissionRate::getRateFor(
                $validated['category_type'],
                $validated['category_id'] ?? null
            );

            if (!$commissionRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'No commission rate found for the specified category'
                ], 404);
            }

            $response = [
                'commission_rate' => $commissionRate,
                'is_currently_effective' => $commissionRate->isCurrentlyEffective()
            ];

            // Calculate commission if total amount is provided
            if (isset($validated['total_amount'])) {
                $response['calculated_commission'] = $commissionRate->calculateCommission($validated['total_amount']);
            }

            return response()->json([
                'success' => true,
                'data' => $response
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get commission rate',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * ✅ NEW: Get commission rate by occupation category ID
     */
    public function getRateByOccupationCategory(Request $request, $categoryId): JsonResponse
    {
        try {
            $commissionRate = CommissionRate::getRateByOccupationCategory($categoryId);

            if (!$commissionRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'No commission rate found for this occupation category'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'commission_rate' => $commissionRate,
                    'is_currently_effective' => $commissionRate->isCurrentlyEffective()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get commission rate',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get currently effective commission rates
     */
    public function getCurrentRates(Request $request): JsonResponse
    {
        try {
            $query = CommissionRate::with(['createdBy', 'occupationCategory'])
                ->active()
                ->currentlyEffective();

            // Filter by applicable_to if provided
            if ($request->has('applicable_to')) {
                $query->where('applicable_to', $request->get('applicable_to'));
            }

            // ✅ NEW: Filter by category_id
            if ($request->has('category_id')) {
                $query->where('category_id', $request->get('category_id'));
            }

            $commissionRates = $query->get();

            return response()->json([
                'success' => true,
                'data' => $commissionRates
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch current commission rates',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
?>
