<?php

namespace App\Http\Controllers;

use App\Models\CommissionEarning;
use App\Models\Booking;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class CommissionEarningController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        try {
            $query = CommissionEarning::with(['booking.parent', 'booking.babysitter', 'booking.payments']);

            // Filter by status
            if ($request->has('status')) {
                $query->where('earning_status', $request->get('status'));
            }

            // Filter by booking_id
            if ($request->has('booking_id')) {
                $query->where('booking_id', $request->get('booking_id'));
            }

            // Filter by date range
            if ($request->has('start_date')) {
                $startDate = Carbon::parse($request->get('start_date'));
                $endDate = $request->has('end_date')
                    ? Carbon::parse($request->get('end_date'))
                    : Carbon::now();

                $query->whereBetween('created_at', [$startDate, $endDate]);
            }

            // Order by
            $orderBy = $request->get('order_by', 'created_at');
            $orderDirection = $request->get('order_direction', 'desc');
            $query->orderBy($orderBy, $orderDirection);

            // Pagination
            $perPage = $request->get('per_page', 10);
            $earnings = $query->paginate($perPage);

            // ✅ FIXED: Use Collection properly
            $earningsData = collect($earnings->items())->map(function($earning) {
                $booking = $earning->booking;
                return [
                    'earning_id' => $earning->earning_id,
                    'booking_id' => $earning->booking_id,
                    'commission_rate' => $earning->commission_rate,
                    'commission_amount' => $earning->commission_amount,
                    'earning_status' => $earning->earning_status,
                    'breakdown' => $booking ? [
                        'total_amount' => $booking->total_amount,
                        'babysitter_earnings' => $booking->babysitter_earnings,
                        'platform_commission' => $earning->commission_amount,
                        'parent_paid' => $booking->total_amount,
                        'babysitter_gets' => $booking->babysitter_earnings
                    ] : null,
                    'booking_details' => $booking ? [
                        'parent_name' => $booking->parent ? $booking->parent->first_name . ' ' . $booking->parent->last_name : null,
                        'babysitter_name' => $booking->babysitter ? $booking->babysitter->first_name . ' ' . $booking->babysitter->last_name : null,
                        'occupation_category' => $booking->occupationCategory ? $booking->occupationCategory->category_name : null,
                        'booking_status' => $booking->booking_status,
                        'payment_status' => $booking->payment_status
                    ] : null,
                    'timeline' => [
                        'created_at' => $earning->created_at,
                        'processed_at' => $earning->processed_at,
                        'paid_out_at' => $earning->paid_out_at
                    ]
                ];
            });

            // Summary statistics
            $totalCommission = $query->sum('commission_amount');
            $pendingCommission = CommissionEarning::pending()->sum('commission_amount');
            $processedCommission = CommissionEarning::processed()->sum('commission_amount');
            $paidOutCommission = CommissionEarning::paidOut()->sum('commission_amount');

            return response()->json([
                'success' => true,
                'data' => $earningsData,
                'summary' => [
                    'total_commission' => (float) $totalCommission,
                    'pending_commission' => (float) $pendingCommission,
                    'processed_commission' => (float) $processedCommission,
                    'paid_out_commission' => (float) $paidOutCommission,
                    'breakdown_explanation' => [
                        'parent_pays_total' => 'Parent pays total booking amount',
                        'platform_commission' => 'Platform deducts commission from total',
                        'babysitter_receives' => 'Babysitter receives total minus commission'
                    ]
                ],
                'pagination' => [
                    'current_page' => $earnings->currentPage(),
                    'per_page' => $earnings->perPage(),
                    'total' => $earnings->total(),
                    'last_page' => $earnings->lastPage(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch commission earnings',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        try {
            // Check if user has permission to create commission earnings
            if (!$request->user()->hasPermission('commission_earnings.create')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }

            $validator = Validator::make($request->all(), [
                'booking_id' => 'required|exists:bookings,booking_id',
                'commission_rate' => 'required|numeric|min:0|max:100',
                'commission_amount' => 'required|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if commission earning already exists for this booking
            $existingEarning = CommissionEarning::where('booking_id', $request->booking_id)->first();
            if ($existingEarning) {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission earning already exists for this booking'
                ], 422);
            }

            $earning = CommissionEarning::create($validator->validated());
            $earning->load(['booking.parent', 'booking.babysitter']);

            return response()->json([
                'success' => true,
                'message' => 'Commission earning created successfully',
                'data' => $earning
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create commission earning',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function show(Request $request, string $id): JsonResponse
    {
        try {
            $earning = CommissionEarning::with(['booking.parent', 'booking.babysitter'])->find($id);

            if (!$earning) {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission earning not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $earning
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch commission earning',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function markAsProcessed(Request $request, string $id): JsonResponse
    {
        try {
            // Check if user has permission to process commission earnings
            if (!$request->user()->hasPermission('commission_earnings.update')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }

            $earning = CommissionEarning::find($id);

            if (!$earning) {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission earning not found'
                ], 404);
            }

            if ($earning->earning_status !== 'pending') {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission earning is not in pending status'
                ], 422);
            }

            $earning->markAsProcessed();

            return response()->json([
                'success' => true,
                'message' => 'Commission earning marked as processed',
                'data' => $earning->load(['booking.parent', 'booking.babysitter'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to process commission earning',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function markAsPaidOut(Request $request, string $id): JsonResponse
    {
        try {
            // Check if user has permission to mark as paid out
            if (!$request->user()->hasPermission('commission_earnings.update')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }

            $earning = CommissionEarning::find($id);

            if (!$earning) {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission earning not found'
                ], 404);
            }

            if ($earning->earning_status !== 'processed') {
                return response()->json([
                    'success' => false,
                    'message' => 'Commission earning must be processed before paying out'
                ], 422);
            }

            $earning->markAsPaidOut();

            return response()->json([
                'success' => true,
                'message' => 'Commission earning marked as paid out',
                'data' => $earning->load(['booking.parent', 'booking.babysitter'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark commission earning as paid out',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function bulkProcess(Request $request): JsonResponse
    {
        try {
            // Check if user has permission to process commission earnings
            if (!$request->user()->hasPermission('commission_earnings.update')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access'
                ], 403);
            }

            $validator = Validator::make($request->all(), [
                'earning_ids' => 'required|array',
                'earning_ids.*' => 'exists:commission_earnings,earning_id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $processedCount = 0;
            $failedCount = 0;

            foreach ($request->earning_ids as $earningId) {
                $earning = CommissionEarning::find($earningId);
                if ($earning && $earning->earning_status === 'pending') {
                    $earning->markAsProcessed();
                    $processedCount++;
                } else {
                    $failedCount++;
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Bulk processing completed',
                'data' => [
                    'processed_count' => $processedCount,
                    'failed_count' => $failedCount,
                    'total_processed' => $processedCount
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to process commission earnings in bulk',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function getMonthlyReport(Request $request): JsonResponse
    {
        try {
            $year = $request->get('year', date('Y'));
            $month = $request->get('month', date('m'));

            $startDate = Carbon::create($year, $month, 1)->startOfMonth();
            $endDate = $startDate->copy()->endOfMonth();

            $earnings = CommissionEarning::with(['booking.parent', 'booking.babysitter'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->get();

            $totalCommission = $earnings->sum('commission_amount');
            $pendingCommission = $earnings->where('earning_status', 'pending')->sum('commission_amount');
            $processedCommission = $earnings->where('earning_status', 'processed')->sum('commission_amount');
            $paidOutCommission = $earnings->where('earning_status', 'paid_out')->sum('commission_amount');

            $categoryBreakdown = $earnings->groupBy(function($earning) {
                return $earning->booking->occupationCategory->category_name ?? 'Unknown';
            })->map(function($categoryEarnings, $categoryName) {
                return [
                    'category' => $categoryName,
                    'total_commission' => $categoryEarnings->sum('commission_amount'),
                    'count' => $categoryEarnings->count()
                ];
            })->values();

            return response()->json([
                'success' => true,
                'data' => [
                    'period' => [
                        'year' => $year,
                        'month' => $month,
                        'start_date' => $startDate->toDateString(),
                        'end_date' => $endDate->toDateString()
                    ],
                    'summary' => [
                        'total_commission' => (float) $totalCommission,
                        'pending_commission' => (float) $pendingCommission,
                        'processed_commission' => (float) $processedCommission,
                        'paid_out_commission' => (float) $paidOutCommission,
                        'total_bookings' => $earnings->count()
                    ],
                    'category_breakdown' => $categoryBreakdown,
                    'earnings' => $earnings
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate monthly report',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * ✅ NEW: Get Commission Breakdown for Specific Booking
     */
    public function getCommissionBreakdown($bookingId): JsonResponse
    {
        try {
            $booking = Booking::with(['parent', 'babysitter', 'occupationCategory', 'payments'])
                ->find($bookingId);

            if (!$booking) {
                return response()->json([
                    'success' => false,
                    'message' => 'Booking not found'
                ], 404);
            }

            $commissionEarning = CommissionEarning::where('booking_id', $bookingId)->first();

            $breakdown = [
                'booking_details' => [
                    'booking_id' => $booking->booking_id,
                    'total_weeks' => $booking->total_weeks,
                    'total_hours' => $booking->total_hours,
                    'hourly_rate' => $booking->agreed_hourly_rate,
                    'parent_name' => $booking->parent ? $booking->parent->first_name . ' ' . $booking->parent->last_name : null,
                    'babysitter_name' => $booking->babysitter ? $booking->babysitter->first_name . ' ' . $booking->babysitter->last_name : null,
                    'occupation_category' => $booking->occupationCategory ? $booking->occupationCategory->category_name : null
                ],
                'financial_breakdown' => [
                    'subtotal_amount' => $booking->subtotal_amount,
                    'total_amount_paid_by_parent' => $booking->total_amount,
                    'commission_rate' => $booking->commission_rate . '%',
                    'commission_amount' => $booking->commission_amount,
                    'babysitter_earnings' => $booking->babysitter_earnings,
                    'calculation' => "Parent pays: {$booking->total_amount}, Platform commission: {$booking->commission_amount}, Babysitter gets: {$booking->babysitter_earnings}"
                ],
                'commission_earning' => $commissionEarning ? [
                    'earning_id' => $commissionEarning->earning_id,
                    'earning_status' => $commissionEarning->earning_status,
                    'processed_at' => $commissionEarning->processed_at,
                    'paid_out_at' => $commissionEarning->paid_out_at
                ] : null,
                'payment_status' => $booking->payment_status,
                'booking_status' => $booking->booking_status
            ];

            return response()->json([
                'success' => true,
                'data' => $breakdown
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch commission breakdown',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * ✅ NEW: Get Commission Earnings by Booking ID
     */
    public function getByBooking($bookingId): JsonResponse
    {
        try {
            $commissionEarnings = CommissionEarning::with(['booking.parent', 'booking.babysitter'])
                ->where('booking_id', $bookingId)
                ->get();

            if ($commissionEarnings->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No commission earnings found for this booking'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $commissionEarnings,
                'summary' => [
                    'total_commission' => $commissionEarnings->sum('commission_amount'),
                    'count' => $commissionEarnings->count()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch commission earnings for booking',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
