<?php

namespace App\Http\Controllers;

use App\Models\BabysitterSkill;
use App\Models\BabysitterProfile;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class BabysitterSkillController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = BabysitterSkill::with(['babysitter']);

            // Filter by babysitter
            if ($request->has('babysitter_id')) {
                $query->where('babysitter_id', $request->get('babysitter_id'));
            }

            // Filter by skill type
            if ($request->has('skill_type')) {
                $query->where('skill_type', $request->get('skill_type'));
            }

            // Filter by verification status
            if ($request->has('verified')) {
                $query->where('is_verified', $request->get('verified'));
            }

            // Order by
            $orderBy = $request->get('order_by', 'skill_name');
            $orderDirection = $request->get('order_direction', 'asc');
            $query->orderBy($orderBy, $orderDirection);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $skills = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $skills->items(),
                'pagination' => [
                    'current_page' => $skills->currentPage(),
                    'per_page' => $skills->perPage(),
                    'total' => $skills->total(),
                    'last_page' => $skills->lastPage(),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch skills',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $skill = BabysitterSkill::with(['babysitter'])->find($id);

            if (!$skill) {
                return response()->json([
                    'success' => false,
                    'message' => 'Skill not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $skill
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch skill',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }



    /**
 * Save/Update babysitter skill with certificate
 */
/**
 * Save/Update babysitter skill with certificate
 */
/**
 * Save/Update babysitter skill with certificate - FIXED VERSION
 */
/**
 * Save/Update babysitter skill with certificate - DEBUG VERSION
 */
public function saveSkill(Request $request): JsonResponse
{
    try {
        $user = $request->user();

        $inputData = $request->all();

        Log::info('=== REQUEST DATA ===');
        Log::info('All data:', $inputData);
        Log::info('Has certificate_image_base64: ' . (isset($inputData['certificate_image_base64']) ? 'YES' : 'NO'));

        if (isset($inputData['certificate_image_base64'])) {
            Log::info('Base64 length: ' . strlen($inputData['certificate_image_base64']));
            Log::info('Base64 starts with: ' . substr($inputData['certificate_image_base64'], 0, 50));
        }

        // ✅ TEMPORARY: Remove certificate_url validation to debug
        $validator = Validator::make($inputData, [
            'babysitter_id' => 'required|exists:babysitter_profiles,profile_id',
            'skill_id' => 'nullable|exists:babysitter_skills,skill_id',
            'skill_type' => 'required|in:certification,skill,language,specialty',
            'skill_name' => 'required|string|max:255',
            'skill_level' => 'nullable|in:beginner,intermediate,advanced,expert',
            'issuing_organization' => 'nullable|string|max:255',
            'issue_date' => 'nullable|date',
            'expiry_date' => 'nullable|date|after:issue_date',
            'delete_certificate' => 'nullable|boolean',
            'certificate_image_base64' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            Log::error('Validation failed:', $validator->errors()->toArray());
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $validated = $validator->validated();

        // Check babysitter profile ownership
        $babysitterProfile = BabysitterProfile::find($validated['babysitter_id']);
        if (!$babysitterProfile) {
            return response()->json([
                'success' => false,
                'message' => 'Babysitter profile not found'
            ], 404);
        }

        if ($babysitterProfile->user_id !== $user->user_id && !$user->isAdmin()) {
            return response()->json([
                'success' => false,
                'message' => 'You can only add skills to your own profile'
            ], 403);
        }

        $certificateUrl = null;

        // ✅ 1. FIRST PRIORITY: Handle base64 image
        if (!empty($inputData['certificate_image_base64'])) {
            Log::info('=== PROCESSING BASE64 ===');

            $base64Image = $inputData['certificate_image_base64'];
            Log::info('Base64 received, checking format...');

            // Check if it's a valid base64 image format
            if (preg_match('/^data:image\/(\w+);base64,/', $base64Image, $type)) {
                Log::info('Valid base64 format detected. Type: ' . $type[1]);

                $imageData = substr($base64Image, strpos($base64Image, ',') + 1);
                $imageType = strtolower($type[1]);

                Log::info('Image type: ' . $imageType);
                Log::info('Image data length: ' . strlen($imageData));

                // Validate image type
                if (!in_array($imageType, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                    Log::error('Invalid image type: ' . $imageType);
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid image format. Only JPG, PNG, GIF, WebP allowed.'
                    ], 422);
                }

                // Decode base64
                Log::info('Decoding base64...');
                $imageData = base64_decode($imageData);

                if ($imageData === false) {
                    Log::error('Base64 decode failed');
                    return response()->json([
                        'success' => false,
                        'message' => 'Base64 decode failed'
                    ], 422);
                }

                Log::info('Base64 decoded successfully. Data length: ' . strlen($imageData));

                // Check file size (max 5MB)
                if (strlen($imageData) > 5 * 1024 * 1024) {
                    Log::error('Image too large: ' . strlen($imageData) . ' bytes');
                    return response()->json([
                        'success' => false,
                        'message' => 'Image size too large. Maximum 5MB allowed.'
                    ], 422);
                }

                // Generate unique filename
                $fileName = time() . '_' . uniqid() . '.' . $imageType;
                $filePath = public_path('storage/skill-certificates/' . $fileName);

                Log::info('File path: ' . $filePath);

                // Ensure directory exists
                $directory = public_path('storage/skill-certificates');
                if (!file_exists($directory)) {
                    Log::info('Creating directory: ' . $directory);
                    mkdir($directory, 0755, true);
                }

                // Save the file
                Log::info('Saving file...');
                if (file_put_contents($filePath, $imageData)) {
                    $certificateUrl = 'skill-certificates/' . $fileName; // ✅ Store FILE PATH, not base64
                    Log::info('✅ Base64 image saved successfully as: ' . $certificateUrl);

                    // Verify file was created
                    if (file_exists($filePath)) {
                        Log::info('✅ File verified: ' . $filePath);
                    } else {
                        Log::error('❌ File not created: ' . $filePath);
                    }
                } else {
                    Log::error('❌ Failed to save image file');
                    return response()->json([
                        'success' => false,
                        'message' => 'Failed to save image file'
                    ], 500);
                }
            } else {
                Log::error('❌ Invalid base64 format');
                Log::info('Base64 starts with: ' . substr($base64Image, 0, 100));
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid base64 format. Must start with data:image/'
                ], 422);
            }
        } else {
            Log::info('No certificate_image_base64 provided');
        }

        // ✅ 2. SECOND: Handle direct URL (only short URLs)
        if (!$certificateUrl && !empty($inputData['certificate_url']) && strlen($inputData['certificate_url']) <= 500) {
            Log::info('Using direct URL: ' . $inputData['certificate_url']);
            $certificateUrl = $inputData['certificate_url'];
        }

        // ✅ 3. THIRD: Handle file upload (form-data)
        if (!$certificateUrl && $request->hasFile('certificate_image')) {
            Log::info('Processing file upload...');
            $file = $request->file('certificate_image');
            $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('storage/skill-certificates'), $fileName);
            $certificateUrl = 'skill-certificates/' . $fileName;
        }

        Log::info('=== FINAL CERTIFICATE URL ===');
        Log::info('Certificate URL to store: ' . ($certificateUrl ?? 'NULL'));

        // Check if updating existing skill or creating new
        if (!empty($validated['skill_id'])) {
            // UPDATE EXISTING SKILL
            $babysitterSkill = BabysitterSkill::where('skill_id', $validated['skill_id'])
                ->where('babysitter_id', $validated['babysitter_id'])
                ->first();

            if (!$babysitterSkill) {
                return response()->json([
                    'success' => false,
                    'message' => 'Skill not found or access denied'
                ], 404);
            }

            // Handle certificate deletion
            if (isset($validated['delete_certificate']) && $validated['delete_certificate']) {
                // Delete file only if it's stored locally
                if ($babysitterSkill->certificate_url && !str_starts_with($babysitterSkill->certificate_url, 'http')) {
                    $filePath = str_replace(url('storage/'), '', $babysitterSkill->certificate_url);
                    if (file_exists(public_path('storage/' . $filePath))) {
                        unlink(public_path('storage/' . $filePath));
                    }
                }
                $certificateUrl = null;
            } elseif (!$certificateUrl) {
                // Keep existing certificate if no new one provided
                $certificateUrl = $babysitterSkill->getRawOriginal('certificate_url');
            }

            $babysitterSkill->update([
                'skill_type' => $validated['skill_type'],
                'skill_name' => $validated['skill_name'],
                'skill_level' => $validated['skill_level'] ?? null,
                'issuing_organization' => $validated['issuing_organization'] ?? null,
                'issue_date' => $validated['issue_date'] ?? null,
                'expiry_date' => $validated['expiry_date'] ?? null,
                'certificate_url' => $certificateUrl
            ]);

            $message = 'Skill updated successfully';
            $statusCode = 200;

        } else {
            // CREATE NEW SKILL
            Log::info('Creating new skill with certificate_url: ' . ($certificateUrl ?? 'NULL'));

            $babysitterSkill = BabysitterSkill::create([
                'babysitter_id' => $validated['babysitter_id'],
                'skill_type' => $validated['skill_type'],
                'skill_name' => $validated['skill_name'],
                'skill_level' => $validated['skill_level'] ?? null,
                'certificate_url' => $certificateUrl, // ✅ This should be the file path
                'issuing_organization' => $validated['issuing_organization'] ?? null,
                'issue_date' => $validated['issue_date'] ?? null,
                'expiry_date' => $validated['expiry_date'] ?? null
            ]);

            $message = 'Skill created successfully';
            $statusCode = 201;
        }

        // Refresh to get the complete data with accessor
        $babysitterSkill->refresh();

        Log::info('=== FINAL RESPONSE ===');
        Log::info('Skill created with certificate_url: ' . ($babysitterSkill->certificate_url ?? 'NULL'));

        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => $babysitterSkill->load(['babysitter'])
        ], $statusCode);

    } catch (\Exception $e) {
        Log::error('Error saving babysitter skill: ' . $e->getMessage());
        Log::error('Stack trace: ' . $e->getTraceAsString());

        return response()->json([
            'success' => false,
            'message' => 'Failed to save skill',
            'error' => config('app.debug') ? $e->getMessage() : null
        ], 500);
    }
}

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $user = request()->user();
            $skill = BabysitterSkill::with(['babysitter'])->find($id);

            if (!$skill) {
                return response()->json([
                    'success' => false,
                    'message' => 'Skill not found'
                ], 404);
            }

            // Check if user owns the skill or is admin
            if ($skill->babysitter->user_id !== $user->user_id && !$user->isAdmin()) {
                return response()->json([
                    'success' => false,
                    'message' => 'You can only delete your own skills'
                ], 403);
            }

            // Delete certificate file if exists
            if ($skill->certificate_url) {
                $filePath = str_replace(url('storage/'), '', $skill->certificate_url);
                if (file_exists(public_path('storage/' . $filePath))) {
                    unlink(public_path('storage/' . $filePath));
                }
            }

            $skill->delete();

            return response()->json([
                'success' => true,
                'message' => 'Skill deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete skill',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Delete only certificate from skill (keep skill record)
     */
    public function deleteCertificate(string $id): JsonResponse
    {
        try {
            $user = request()->user();
            $babysitterSkill = BabysitterSkill::with(['babysitter'])->find($id);

            if (!$babysitterSkill) {
                return response()->json([
                    'success' => false,
                    'message' => 'Skill not found'
                ], 404);
            }

            // Check if user owns the skill or is admin
            if ($babysitterSkill->babysitter->user_id !== $user->user_id && !$user->isAdmin()) {
                return response()->json([
                    'success' => false,
                    'message' => 'You can only delete certificates from your own skills'
                ], 403);
            }

            if (!$babysitterSkill->certificate_url) {
                return response()->json([
                    'success' => false,
                    'message' => 'No certificate found to delete'
                ], 404);
            }

            // Delete file from storage
            $filePath = str_replace(url('storage/'), '', $babysitterSkill->certificate_url);
            if (file_exists(public_path('storage/' . $filePath))) {
                unlink(public_path('storage/' . $filePath));
            }

            // Update skill record - remove certificate URL
            $babysitterSkill->update(['certificate_url' => null]);

            return response()->json([
                'success' => true,
                'message' => 'Certificate deleted successfully',
                'data' => $babysitterSkill->fresh()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete certificate',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
