<?php

namespace App\Events;

use App\Models\ParentProfile;
use Illuminate\Broadcasting\Channel;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class ParentProfileCreated implements ShouldBroadcast
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public $profile;

    /**
     * Create a new event instance.
     */
    public function __construct(ParentProfile $profile)
    {
        $this->profile = $profile;
    }

    /**
     * Get the channels the event should broadcast on.
     */
    public function broadcastOn(): array
    {
        return [
            new PrivateChannel('superadmin.notifications'),
        ];
    }

    /**
     * The event's broadcast name.
     */
    public function broadcastAs(): string
    {
        return 'parent.profile.created';
    }

    /**
     * Get the data to broadcast.
     */
    public function broadcastWith(): array
    {
        // Role based user type determine karein
        $userType = 'User';
        if ($this->profile->user->role_id == 2) {
            $userType = 'Parent';
        } elseif ($this->profile->user->role_id == 3) {
            $userType = 'Babysitter';
        }

        return [
            'profile_id' => $this->profile->profile_id,
            'user_id' => $this->profile->user_id,
            'user_name' => $this->profile->user->first_name . ' ' . $this->profile->user->last_name,
            'user_email' => $this->profile->user->email,
            'user_type' => $userType,
            'role_id' => $this->profile->user->role_id,
            'message' => 'New Parent profile created: ' . $this->profile->user->first_name . ' ' . $this->profile->user->last_name,
            'profile_picture' => $this->profile->user->profile_picture_url,
            'family_description' => $this->profile->family_description,
            'preferred_communication' => $this->profile->preferred_communication,
            'care_recipients_count' => $this->profile->careRecipients->count(),
            'registration_date' => $this->profile->created_at->toDateTimeString(),
            'timestamp' => now()->toDateTimeString(),
            'icon' => '👨‍👩‍👧‍👦',
            'action_url' => '/admin/parents/' . $this->profile->profile_id
        ];
    }
}
