<?php

namespace App\Events;

use App\Models\DirectBookingRequest;
use Illuminate\Broadcasting\Channel;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class NewDirectBookingRequest implements ShouldBroadcast
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public $bookingRequest;

    /**
     * Create a new event instance.
     */
    public function __construct(DirectBookingRequest $bookingRequest)
    {
        $this->bookingRequest = $bookingRequest;
    }

    /**
     * Get the channels the event should broadcast on.
     */
    public function broadcastOn(): array
    {
        // Babysitter ke private channel par broadcast karein
        return [
            new PrivateChannel('user.' . $this->bookingRequest->babysitter_id . '.notifications'),
        ];
    }

    /**
     * The event's broadcast name.
     */
    public function broadcastAs(): string
    {
        return 'direct.booking.request.created';
    }

    /**
     * Get the data to broadcast.
     */
    public function broadcastWith(): array
    {
        return [
            'booking_request_id' => $this->bookingRequest->booking_request_id,
            'parent_id' => $this->bookingRequest->parent_id,
            'babysitter_id' => $this->bookingRequest->babysitter_id,
            'parent_name' => $this->bookingRequest->parent->first_name . ' ' . $this->bookingRequest->parent->last_name,
            'parent_email' => $this->bookingRequest->parent->email,
            'start_date' => $this->bookingRequest->start_date->toDateString(),
            'end_date' => $this->bookingRequest->end_date ? $this->bookingRequest->end_date->toDateString() : null,
            'expected_hours_per_week' => $this->bookingRequest->expected_hours_per_week,
            'hourly_rate' => $this->bookingRequest->hourly_rate,
            'total_expected_amount' => $this->bookingRequest->total_expected_amount,
            'care_type' => $this->bookingRequest->care_type,
            'number_of_recipients' => $this->bookingRequest->number_of_recipients,
            'request_status' => $this->bookingRequest->request_status,
            'payment_status' => $this->bookingRequest->payment_status,
            'expires_at' => $this->bookingRequest->expires_at->toDateTimeString(),
            'remaining_hours' => $this->bookingRequest->getRemainingHours(),
            'message' => 'New booking request from ' . $this->bookingRequest->parent->first_name . ' ' . $this->bookingRequest->parent->last_name,
            'timestamp' => now()->toDateTimeString(),
            'icon' => '📅',
            'action_url' => '/booking-requests/' . $this->bookingRequest->booking_request_id
        ];
    }
}
